/**
 * This file is part of Totara Enterprise Extensions.
 *
 * Copyright (C) 2020 onwards Totara Learning Solutions LTD
 *
 * Totara Enterprise Extensions is provided only to Totara
 * Learning Solutions LTD's customers and partners, pursuant to
 * the terms and conditions of a separate agreement with Totara
 * Learning Solutions LTD or its affiliate.
 *
 * If you do not have an agreement with Totara Learning Solutions
 * LTD, you may not access, use, modify, or distribute this software.
 * Please contact [licensing@totaralearning.com] for more information.
 *
 * @author Simon Chester <simon.chester@totaralearning.com>
 * @module editor_weka
 */

import tui from 'tui/tui';
import { throttle, createNewEvent, uniqueId } from 'tui/util';
import Weka from 'editor_weka/components/Weka';
import WekaValue from './WekaValue';

/**
 * Set up Weka editor on element generated by core editor code.
 *
 * @param {object} options
 * @param {string} options.id             ID of textarea in DOM
 * @param {object[]} options.extensions
 * @param {boolean} options.showtoolbar
 * @param {number} options.file_item_id   The file draft's item id.
 * @param {number} options.context_id     The context's id.
 * @param {object[]} options.files        The array of current files within the specific area.
 */
export function setupTextarea({
  id,
  extensions,
  showtoolbar,
  file_item_id,
  context_id,
  files,
}) {
  const el = document.getElementById(id);
  if (!el) {
    const error = new Error('Textarea not found');
    error.code = 'TEXTAREA_NOT_FOUND';
    throw error;
  }

  const options = {
    extensions,
    showtoolbar,
    context_id,
    files,
    file_item_id: file_item_id,
  };

  new IntegrationView(id, el, options);
}

class IntegrationView {
  constructor(id, textareaEl, editorOptions) {
    this.id = id;
    this.textareaEl = textareaEl;
    textareaEl.style.display = 'none';
    textareaEl.setAttribute('data-enhanced', true);

    const label = document.querySelector(`label[for="${CSS.escape(id)}"]`);
    let labelId = null;
    if (label) {
      if (!label.id) {
        label.id = 'uid-' + uniqueId();
      }
      labelId = label.id;
    }

    this.wrapper = document.createElement('div');

    this.textareaEl.parentNode.insertBefore(
      this.wrapper,
      this.textareaEl.nextSibling
    );

    let initialValue = this.getWekaValue();
    this.wekaValue = initialValue;

    const update = () => {
      if (this.wekaValue) {
        let event = createNewEvent('change');
        this.textareaEl.value = JSON.stringify(this.wekaValue.getDoc());
        this.textareaEl.dispatchEvent(event);
      }
    };
    const throttledUpdate = throttle(update, 250);

    tui.mount(
      Weka,
      {
        variant: 'full',
        value: initialValue,
        options: editorOptions,
        fileItemId: editorOptions.file_item_id
          ? editorOptions.file_item_id
          : undefined,
        ariaLabelledby: labelId,
        onInput: value => {
          this.wekaValue = value;
          throttledUpdate();
        },
        onBlur: update,
        onReady: () => {
          if (this.wekaValue.hasHtml()) {
            update();
          }
        },
      },
      this.wrapper
    );
  }

  /**
   * Get document from textarea
   *
   * @returns object
   */
  getWekaValue() {
    let rawVal = this.textareaEl.value;

    if (rawVal) {
      if (this.looksLikeJson(rawVal)) {
        try {
          const doc = JSON.parse(rawVal);
          return WekaValue.fromDoc(doc);
        } catch (e) {
          console.error('[editor_weka] Error parsing JSON');
          console.error(e);
        }
      } else {
        return WekaValue.fromHtml(rawVal);
      }
    }

    return WekaValue.empty();
  }

  looksLikeJson(str) {
    return str.slice(0, 1) === '{' && str.slice(str.length - 1) === '}';
  }
}
