/**
 * This file is part of Totara Enterprise Extensions.
 *
 * Copyright (C) 2024 onwards Totara Learning Solutions LTD
 *
 * Totara Enterprise Extensions is provided only to Totara
 * Learning Solutions LTD's customers and partners, pursuant to
 * the terms and conditions of a separate agreement with Totara
 * Learning Solutions LTD or its affiliate.
 *
 * If you do not have an agreement with Totara Learning Solutions
 * LTD, you may not access, use, modify, or distribute this software.
 * Please contact [licensing@totara.com] for more information.
 *
 * @author Simon Chester <simon.chester@totara.com>
 * @module tui
 */

import { formatParams } from 'tui/util';
import { waitFor } from 'tui_test_utils/vtl';
import {
  notify,
  notifyParams,
  showPendingNotifications,
  testInternals,
} from '../notifications';

let container = null;

jest.mock('../tui', () => {
  return {
    mount(comp, props) {
      const c = {
        addNotification: jest.fn(),
      };
      container = c;
      props.ref.value = c;
    },
  };
});

jest.mock('../storage');

beforeEach(() => {
  testInternals.storage().clear();
  if (container) {
    container.addNotification.mockReset();
  }
});

describe('notify', () => {
  it('calls addNotification on container', async () => {
    await notify({ message: 'hi' });
    expect(container.addNotification).toHaveBeenCalledOnceWith({
      duration: 5000,
      message: 'hi',
      type: 'success',
    });
  });
});

describe('notifyParams', () => {
  it('provides one time use code and notification data as URL params', () => {
    const params = notifyParams({ message: 'foobar' });
    expect(Object.keys(params)).toEqual(['nf', 'nfd']);
    const token = testInternals.storage().get('token');
    expect(token).not.toBeNull();
    expect(params.nf).toBe(token);
    expect(JSON.parse(decodeURIComponent(atob(params.nfd)))).toEqual({
      message: 'foobar',
    });
  });

  it('supports multi-byte text', () => {
    const params = notifyParams({ message: 'いい天気' });
    expect(JSON.parse(decodeURIComponent(atob(params.nfd)))).toEqual({
      message: 'いい天気',
    });
  });
});

describe('showPendingNotifications', () => {
  it('shows notifications from URL', async () => {
    Object.defineProperty(window, 'location', {
      configurable: true,
      value: new URL(
        'http://localhost/?' + formatParams(notifyParams({ message: 'hello' }))
      ),
    });
    showPendingNotifications();
    waitFor(() => {
      expect(container.addNotification).toHaveBeenCalledOnceWith({
        duration: 5000,
        message: 'hello',
        type: 'success',
      });
    });
  });

  it("does not show anything if the URL params aren't present", async () => {
    Object.defineProperty(window, 'location', {
      configurable: true,
      value: new URL('http://localhost/'),
    });
    showPendingNotifications();
    waitFor(() => {
      expect(container.addNotification).not.toHaveBeenCalled();
    });
  });

  it('does not show anything if the URL params are incorrect', async () => {
    Object.defineProperty(window, 'location', {
      configurable: true,
      value: new URL(
        'http://localhost/?' +
          formatParams({ ...notifyParams({ message: 'hello' }), nf: '0' })
      ),
    });
    showPendingNotifications();
    waitFor(() => {
      expect(container.addNotification).not.toHaveBeenCalled();
    });
  });

  it('supports multi-byte text', async () => {
    Object.defineProperty(window, 'location', {
      configurable: true,
      value: new URL(
        'http://localhost/?' +
          formatParams(notifyParams({ message: 'いい天気' }))
      ),
    });
    showPendingNotifications();
    waitFor(() => {
      expect(container.addNotification).toHaveBeenCalledOnceWith({
        duration: 5000,
        message: 'いい天気',
        type: 'success',
      });
    });
  });
});
