/**
 * This file is part of Totara Enterprise Extensions.
 *
 * Copyright (C) 2020 onwards Totara Learning Solutions LTD
 *
 * Totara Enterprise Extensions is provided only to Totara
 * Learning Solutions LTD's customers and partners, pursuant to
 * the terms and conditions of a separate agreement with Totara
 * Learning Solutions LTD or its affiliate.
 *
 * If you do not have an agreement with Totara Learning Solutions
 * LTD, you may not access, use, modify, or distribute this software.
 * Please contact [licensing@totara.com] for more information.
 *
 * @author Simon Chester <simon.chester@totara.com>
 * @module tui
 */

const path = require('path');
const fs = require('fs');
const prettier = require('prettier');
const { rootDir } = require('../lib/common');
const { dirMaps } = require('../lib/resolution');
const scanComponents = require('../webpack/scan_components');
const prettierConfig = require(path.join(rootDir, '.prettierrc'));

/**
 * Stringify the provided JSON, pass it through Prettier, and write it to the provided file path.
 *
 * @param {string} path
 * @param {*} json
 */
function writeJson(path, json) {
  let content = JSON.stringify(json, null, 2);
  content = prettier.format(content, { ...prettierConfig, parser: 'json' });
  fs.writeFileSync(path, content, 'utf8');
}

/**
 * Generate project configs for editor support
 */
function generateProjectConfigs() {
  const components = scanComponents({ rootDir });
  generateJsconfig(components);
}

/**
 * Generate jsconfig.json (VS Code)
 *
 * @param {import('../webpack/scan_components').ComponentInfo[]} components
 */
function generateJsconfig(components) {
  const paths = {};

  for (const comp of components) {
    if (comp.structureVersion === 1) {
      dirMaps.forEach(dirMap => {
        paths[comp.name + dirMap.idBaseSuffix + '/*'] = [
          './' + path.join(comp.path, 'src', dirMap.path) + '/*',
        ];
      });
    } else if (comp.structureVersion === 2) {
      paths[comp.name + '/*'] = [`./${comp.path}/src/*`, `./${comp.path}/src/js/*`];
    }
  }

  paths['tui_test_utils/*'] = ['./client/tooling/jest/tui_test_utils/*'];

  const jsconfig = {
    '//': 'generated by `npm run generate-configs`',
    compilerOptions: {
      baseUrl: '.',
      paths,
    },
    include: ['client/component/*/src'],
    exclude: ['node_modules', 'vendor', 'build'],
  };

  writeJson(path.join(rootDir, 'jsconfig.json'), jsconfig);
}

generateProjectConfigs();
