<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

namespace core_badges\model;

use core\entity\user;
use core\orm\entity\encrypted_model;
use core_badges\entity\badge_email_verify as badge_email_verify_entity;

/**
 * Badge email verify model class
 *
 * Properties:
 * @property-read int $id
 * @property-read int $user_id
 * @property-read string $secret
 * @property-read string $address
 * @property-read string $backpackid
 * @property-read string $password
 */
class badge_email_verify extends encrypted_model {

    /** @var badge_email_verify_entity */
    protected $entity;

    /** @var string[] */
    protected $entity_attribute_whitelist = [
        'id',
        'user_id',
        'secret',
        'address',
        'backpackid',
        'password',
    ];

    /** @var string[] */
    protected $model_accessor_whitelist = [];

    /** @var string[] */
    protected $encrypted_attribute_list = [
        'secret',
        'password',
    ];

    /**
     * @inheritDoc
     */
    protected static function get_entity_class(): string {
        return badge_email_verify_entity::class;
    }

    /**
     * Creates a new badge_email_verify record or if it already exists for
     * that user_id, will update the record.
     *
     * @param string $address
     * @param string $backpackid
     * @param string $password unencrypted
     * @param string $secret
     *
     * @return badge_email_verify|null
     */
    public static function create(
        string $address,
        string $backpackid,
        string $password,
        string $secret,
    ): ?badge_email_verify {

        $user = user::logged_in();
        if (!$user) {
            // Can only work with logged-in user.
            return null;
        }

        // If a record already exists for this user: update it.
        $entity = badge_email_verify_entity::repository()
            ->where('user_id', $user->id)
            ->get()
            ->first();

        if (!$entity) {
            $entity = new badge_email_verify_entity();
        }

        $entity->user_id = $user->id;
        $entity->address = $address;
        $entity->backpackid = $backpackid;
        $entity->password = ''; // Encrypted value will be set below
        $entity->secret = ''; // Encrypted value will be set below
        $entity->save();

        // Need to set encrypted values after saving the entity.
        $model = static::load_by_entity($entity);
        $model->set_encrypted_attribute('password', $password);
        $model->set_encrypted_attribute('secret', $secret);

        // There is a possibility that previous values are still in the user_preferences, so make sure they are removed there.
        // This can happen if there was a pending verification before upgrade, but the user triggers a new verification after upgrade.
        unset_user_preference('badges_email_verify_secret');
        unset_user_preference('badges_email_verify_address');
        unset_user_preference('badges_email_verify_backpackid');
        unset_user_preference('badges_email_verify_password');

        return $model;
    }

    /**
     * @return static|null
     */
    public static function for_logged_in_user(): ?static {
        $entity = static::get_entity_for_logged_in_user();
        return $entity ? static::load_by_entity($entity) : null;
    }

    /**
     * @return badge_email_verify_entity|null
     */
    private static function get_entity_for_logged_in_user(): ?badge_email_verify_entity {
        $user = user::logged_in();
        if (!$user) {
            // Can only work with logged-in user.
            return null;
        }

        return badge_email_verify_entity::repository()
            ->where('user_id', $user->id)
            ->get()
            ->first();
    }

    /**
     * @return string|null
     */
    public static function fetch_secret(): ?string {
        return static::fetch_value('secret');
    }

    /**
     * @return string|null
     */
    public static function fetch_address(): ?string {
        return static::fetch_value('address');
    }

    /**
     * It's fine to return the backpack ID as string.
     * In fact, it preserves the old behaviour when only get_user_preferences() was used for this.
     *
     * @return string|null
     */
    public static function fetch_backpackid(): ?string {
        return static::fetch_value('backpackid');
    }

    /**
     * @return string|null
     */
    public static function fetch_password(): ?string {
        return static::fetch_value('password');
    }

    /**
     * Get the property value for the given key.
     * If there is no record for this user, we try to find the value in the user_preferences table,
     * which is where it was stored before we introduced the badge_email_verify table. This is to make
     * sure a pending verification still works after upgrade.
     *
     * @param string $key
     * @return string|null
     */
    private static function fetch_value(string $key): ?string {
        $model = static::for_logged_in_user();
        if ($model) {
            $value = $model->{$key};

            // The decryption method will return false if the column has a null value. In this case we want
            // to return null here to reflect the actual DB value.
            if ($value === false) {
                return null;
            }
            return $value;
        }

        // There is a possibility that the value is still in the user_preferences.
        return get_user_preferences('badges_email_verify_' . $key);
    }

    /**
     * The verification process needs to unset the secret separately, so we need an extra method for that.
     *
     * @return void
     */
    public static function unset_secret(): void {
        $entity = static::get_entity_for_logged_in_user();
        if ($entity) {
            $entity->secret = null;
            $entity->save();
        }

        // There is a possibility that the value is still in the user_preferences,
        // so make sure it's removed there as well.
        unset_user_preference('badges_email_verify_secret');
    }

    /**
     * The verification process needs to unset the email address separately, so we need an extra method for that.
     *
     * @return void
     */
    public static function unset_address(): void {
        $entity = static::get_entity_for_logged_in_user();
        if ($entity) {
            $entity->address = null;
            $entity->save();
        }

        // There is a possibility that the value is still in the user_preferences,
        // so make sure it's removed there as well.
        unset_user_preference('badges_email_verify_address');
    }

    /**
     * Unset all the values.
     * This means deleting the entity and making sure no leftover data is in the user_preferences.
     *
     * @return void
     */
    public static function unset_all(): void {
        $entity = static::get_entity_for_logged_in_user();
        if ($entity) {
            $entity->delete();
        }

        // There is a possibility that the values are still in the user_preferences,
        // so make sure it's removed there as well.
        unset_user_preference('badges_email_verify_secret');
        unset_user_preference('badges_email_verify_address');
        unset_user_preference('badges_email_verify_backpackid');
        unset_user_preference('badges_email_verify_password');
    }
}