<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

use core_badges\entity\badge_email_verify as badge_email_verify_entity;
use core_badges\model\badge_email_verify;
use core_phpunit\testcase;

class core_badges_badge_email_verify_model_test extends testcase {

    public function test_create_new_record() {
        static::setUser(null);

        // Should return null if we don't have a logged-in user.
        $model = badge_email_verify::create(
            'test@example.com',
            '123',
            'my_password',
            'my_secret',
        );
        static::assertNull($model);

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        static::assertSame(0, badge_email_verify_entity::repository()->count());

        /** @var badge_email_verify $model */
        $model = badge_email_verify::create(
            'test@example.com',
            '123',
            'my_password',
            'my_secret',
        );

        static::assertSame(1, badge_email_verify_entity::repository()->count());

        $entity = badge_email_verify_entity::repository()->get()->first();

        static::assertEquals($user->id, $entity->user_id);
        static::assertSame('test@example.com', $entity->address);
        static::assertSame('123', $entity->backpackid);

        // Should not be saved in plain text.
        static::assertNotEquals('my_password', $entity->password);
        static::assertNotEmpty($entity->password);
        // Should not be saved in plain text.
        static::assertNotEquals('my_secret', $entity->secret);
        static::assertNotEmpty($entity->secret);

        // The model should return password and secret in plain text.
        static::assertEquals($user->id, $model->user_id);
        static::assertSame('my_password', $model->password);
        static::assertSame('my_secret', $model->secret);
        static::assertSame('test@example.com', $model->address);
        static::assertSame('123', $model->backpackid);
    }

    public function test_create_updates_existing_record() {
        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        badge_email_verify::create(
            'first_save@example.com',
            '789',
            'first_save_password',
            'first_save_secret',
        );

        // Calling badge_email_verify::create() again for the same user will overwrite the existing values.
        // There can be only one record for one user.
        /** @var badge_email_verify $model */
        $model = badge_email_verify::create(
            'test@example.com',
            '123',
            'my_password',
            'my_secret',
        );

        static::assertSame(1, badge_email_verify_entity::repository()->count());

        $entity = badge_email_verify_entity::repository()->get()->first();

        // Make sure the record got updated with all the new data.
        static::assertEquals($user->id, $entity->user_id);
        static::assertSame('test@example.com', $entity->address);
        static::assertSame('123', $entity->backpackid);

        // Should not be saved in plain text.
        static::assertNotEquals('my_password', $entity->password);
        static::assertNotEmpty($entity->password);
        // Should not be saved in plain text.
        static::assertNotEquals('my_secret', $entity->secret);
        static::assertNotEmpty($entity->secret);

        // The model should return password and secret in plain text.
        static::assertEquals($user->id, $model->user_id);
        static::assertSame('my_password', $model->password);
        static::assertSame('my_secret', $model->secret);
        static::assertSame('test@example.com', $model->address);
        static::assertSame('123', $model->backpackid);
    }

    public function test_create_removes_legacy_pending_values(): void {
        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Set legacy values in user_preferences table.
        set_user_preference('badges_email_verify_secret', 'my_old_secret');
        set_user_preference('badges_email_verify_backpackid', '111');
        set_user_preference('badges_email_verify_password', 'my_old_password');
        set_user_preference('badges_email_verify_address', 'old_email@example.com');

        /** @var badge_email_verify $model */
        $model = badge_email_verify::create(
            'test@example.com',
            '999',
            'my_password',
            'my_secret',
        );

        // Legacy values should have been removed from the user_preferences.
        static::assertNull(get_user_preferences('badges_email_verify_secret'));
        static::assertNull(get_user_preferences('badges_email_verify_backpackid'));
        static::assertNull(get_user_preferences('badges_email_verify_password'));
        static::assertNull(get_user_preferences('badges_email_verify_address'));

        // Make sure the model was still created with new values.
        static::assertEquals($user->id, $model->user_id);
        static::assertSame('my_password', $model->password);
        static::assertSame('my_secret', $model->secret);
        static::assertSame('test@example.com', $model->address);
        static::assertSame('999', $model->backpackid);
    }

    public function test_fetch_secret(): void {
        static::setUser(null);

        // Should return null if we don't have a logged-in user.
        static::assertNull(badge_email_verify::fetch_secret());

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should return null if nothing is set in DB.
        static::assertNull(badge_email_verify::fetch_secret());

        // Should return user_preferences value if only that one is set.
        set_user_preference('badges_email_verify_secret', 'my_old_secret');
        static::assertSame('my_old_secret', badge_email_verify::fetch_secret());

        // Should return value from badge_email_verify table if both are set.
        $model = badge_email_verify::create(
            'test@example.com',
            '123',
            'my_password',
            'my_new_secret',
        );
        static::assertSame('my_new_secret', badge_email_verify::fetch_secret());

        // Should return value from badge_email_verify table if only that one is set.
        unset_user_preference('badges_email_verify_secret');
        static::assertSame('my_new_secret', badge_email_verify::fetch_secret());

        // The secret is encrypted in DB. If it's in plain text for any reason, it should still work - the plain text should be returned.
        $entity = badge_email_verify_entity::repository()->get()->first();
        static::assertNotEquals('my_new_secret', $entity->secret);
        $entity->secret = 'my_plain_text_secret';
        $entity->save();
        static::assertSame('my_plain_text_secret', badge_email_verify::fetch_secret());
    }

    public function test_fetch_password(): void {
        static::setUser(null);

        // Should return null if we don't have a logged-in user.
        static::assertNull(badge_email_verify::fetch_password());

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should return null if nothing is set in DB.
        static::assertNull(badge_email_verify::fetch_password());

        // Should return user_preferences value if only that one is set.
        set_user_preference('badges_email_verify_password', 'my_old_password');
        static::assertSame('my_old_password', badge_email_verify::fetch_password());

        // Should return value from badge_email_verify table if both are set.
        $model = badge_email_verify::create(
            'test@example.com',
            '123',
            'my_new_password',
            'my_secret',
        );
        static::assertSame('my_new_password', badge_email_verify::fetch_password());

        // Should return value from badge_email_verify table if only that one is set.
        unset_user_preference('badges_email_verify_password');
        static::assertSame('my_new_password', badge_email_verify::fetch_password());

        // The password is encrypted in DB. If it's in plain text for any reason, it should still work - the plain text should be returned.
        $entity = badge_email_verify_entity::repository()->get()->first();
        static::assertNotEquals('my_new_password', $entity->password);
        $entity->password = 'my_plain_text_password';
        $entity->save();
        static::assertSame('my_plain_text_password', badge_email_verify::fetch_password());
    }

    public function test_fetch_address(): void {
        static::setUser(null);

        // Should return null if we don't have a logged-in user.
        static::assertNull(badge_email_verify::fetch_address());

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should return null if nothing is set in DB.
        static::assertNull(badge_email_verify::fetch_address());

        // Should return user_preferences value if only that one is set.
        set_user_preference('badges_email_verify_address', 'my_old_address@example.com');
        static::assertSame('my_old_address@example.com', badge_email_verify::fetch_address());

        // Should return value from badge_email_verify table if both are set.
        $model = badge_email_verify::create(
            'my_new_address@example.com',
            '123',
            'my_password',
            'my_secret',
        );
        static::assertSame('my_new_address@example.com', badge_email_verify::fetch_address());

        // Should return value from badge_email_verify table if only that one is set.
        unset_user_preference('badges_email_verify_address');
        static::assertSame('my_new_address@example.com', badge_email_verify::fetch_address());
    }

    public function test_fetch_backpackid(): void {
        static::setUser(null);

        // Should return null if we don't have a logged-in user.
        static::assertNull(badge_email_verify::fetch_backpackid());

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should return null if nothing is set in DB.
        static::assertNull(badge_email_verify::fetch_backpackid());

        // Should return user_preferences value if only that one is set.
        set_user_preference('badges_email_verify_backpackid', 111);
        static::assertSame('111', badge_email_verify::fetch_backpackid());

        // Should return value from badge_email_verify table if both are set.
        $model = badge_email_verify::create(
            'test@example.com',
            '999',
            'my_password',
            'my_secret',
        );
        static::assertSame('999', badge_email_verify::fetch_backpackid());

        // Should return value from badge_email_verify table if only that one is set.
        unset_user_preference('badges_email_verify_backpackid');
        static::assertSame('999', badge_email_verify::fetch_backpackid());
    }

    public function test_unset_secret(): void {
        static::setUser(null);

        // Shouldn't fail if there's no logged-in user.
        badge_email_verify::unset_secret();

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should remove the entry in user_preferences if there is one.
        set_user_preference('badges_email_verify_secret', 'my_secret');
        static::assertSame('my_secret', get_user_preferences('badges_email_verify_secret'));
        badge_email_verify::unset_secret();
        static::assertNull(get_user_preferences('badges_email_verify_secret'));
        static::assertNull(badge_email_verify::fetch_secret());

        $model = badge_email_verify::create(
            'test@example.com',
            '999',
            'my_password',
            'my_new_secret',
        );
        // Set a legacy value in user_preferences as well.
        set_user_preference('badges_email_verify_secret', 'my_old_secret');
        static::assertSame('my_new_secret', badge_email_verify::fetch_secret());

        badge_email_verify::unset_secret();

        // 'secret' column should be null now
        static::assertNull(badge_email_verify::fetch_secret());
        // Make sure that the legacy value in user_preferences has also been removed.
        static::assertNull(get_user_preferences('badges_email_verify_secret'));

        // Make sure the entity is still there.
        /** @var badge_email_verify_entity $entity */
        $entity = badge_email_verify_entity::repository()->one(true);
        static::assertNotNull($entity->password);
        static::assertNotNull($entity->address);
        static::assertNotNull($entity->backpackid);
        static::assertNull($entity->secret);
    }
    
    public function test_unset_address(): void {
        static::setUser(null);

        // Shouldn't fail if there's no logged-in user.
        badge_email_verify::unset_address();

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Should remove the entry in user_preferences if there is one.
        set_user_preference('badges_email_verify_address', 'my_address@example.com');
        static::assertSame('my_address@example.com', get_user_preferences('badges_email_verify_address'));
        badge_email_verify::unset_address();
        static::assertNull(get_user_preferences('badges_email_verify_address'));
        static::assertNull(badge_email_verify::fetch_address());

        $model = badge_email_verify::create(
            'my_new_address@example.com',
            '999',
            'my_password',
            'my_secret',
        );
        // Set a legacy value in user_preferences as well.
        set_user_preference('badges_email_verify_address', 'my_old_address@example.com');
        static::assertSame('my_new_address@example.com', badge_email_verify::fetch_address());

        badge_email_verify::unset_address();

        // 'address' column should be null now
        static::assertNull(badge_email_verify::fetch_address());
        // Make sure that the legacy value in user_preferences has also been removed.
        static::assertNull(get_user_preferences('badges_email_verify_address'));

        // Make sure the entity is still there.
        /** @var badge_email_verify_entity $entity */
        $entity = badge_email_verify_entity::repository()->one(true);
        static::assertNotNull($entity->password);
        static::assertNotNull($entity->secret);
        static::assertNotNull($entity->backpackid);
        static::assertNull($entity->address);
    }

    public function test_unset_all(): void {
        static::setUser(null);

        // Shouldn't fail if there's no logged-in user.
        badge_email_verify::unset_all();

        $other_user = static::getDataGenerator()->create_user();
        static::setUser($other_user);

        // Set values for the badge_email_verify table.
        $model = badge_email_verify::create(
            'my_address@example.com',
            '999',
            'my_password',
            'my_secret',
        );

        $user = static::getDataGenerator()->create_user();
        static::setUser($user);

        // Set values for the badge_email_verify table.
        $model = badge_email_verify::create(
            'my_address@example.com',
            '888',
            'my_password',
            'my_secret',
        );
        static::assertSame(2, badge_email_verify_entity::repository()->count());
        static::assertSame(1, badge_email_verify_entity::repository()->where('user_id', $user->id)->count());
        static::assertSame(1, badge_email_verify_entity::repository()->where('user_id', $other_user->id)->count());

        set_user_preference('badges_email_verify_address', 'my_old_address@example.com');
        set_user_preference('badges_email_verify_secret', 'my_secret');
        set_user_preference('badges_email_verify_password', 'my_password');
        set_user_preference('badges_email_verify_backpackid', '777');

        badge_email_verify::unset_all();

        static::assertSame(1, badge_email_verify_entity::repository()->count());
        static::assertSame(1, badge_email_verify_entity::repository()->where('user_id', $other_user->id)->count());

        static::assertNull(get_user_preferences('badges_email_verify_address'));
        static::assertNull(get_user_preferences('badges_email_verify_secret'));
        static::assertNull(get_user_preferences('badges_email_verify_password'));
        static::assertNull(get_user_preferences('badges_email_verify_backpackid'));
    }
}