<?php
/*
 * This file is part of Totara Learn
 *
 * Copyright (C) 2018 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Sam Hemelryk <sam.hemelryk@totaralearning.com>
 * @package block_admin_related_pages
 */

global $CFG;
require_once($CFG->libdir.'/adminlib.php');

use block_admin_related_pages\group;
use block_admin_related_pages\item;
use block_admin_related_pages\map;

/**
 * Map tests
 *
 * @package block_admin_related_pages
 * @group block_admin_related_pages
 */
class block_admin_related_pages_map_test extends \core_phpunit\testcase {

    protected function tearDown(): void {
        global $ADMIN;
        $ADMIN = null;
        parent::tearDown(); // TODO: Change the autogenerated stub
    }

    /**
     * @param map $map
     * @return \PHPUnit\Framework\MockObject\MockObject|\admin_root
     */
    private function mock_admin_root(map $map) {
        $root = $this->createMock(\admin_root::class);
        $property = new \ReflectionProperty(map::class, 'adminroot');
        $property->setAccessible(true);
        $property->setValue($map, $root);
        return $root;
    }

    /**
     * @param string $key
     * @param string $name
     * @param string $url
     * @return \PHPUnit\Framework\MockObject\MockObject|\admin_externalpage
     */
    private function mock_admin_externalpage(string $key = 'test', string $name = 'Test', string $url = '/test', $can_access = true, $hidden = false) {
        $page = $this->getMockBuilder(\admin_externalpage::class)
            ->onlyMethods(['check_access', 'is_hidden'])
            ->setConstructorArgs([$key, $name, $url])
            ->getMock();
        $page->expects($this->any())->method('check_access')->willReturn($can_access);
        $page->expects($this->any())->method('is_hidden')->willReturn($hidden);
        return $page;
    }

    private function mock_admin_category(string $name = 'test', string $visiblename = 'Test', $pages = [], $can_access = true, $hidden = false) {
        $category = $this->getMockBuilder(\admin_category::class)
            ->onlyMethods(['check_access', 'is_hidden', 'get_children'])
            ->setConstructorArgs([$name, $visiblename, $hidden])
            ->getMock();
        $category->expects($this->any())->method('get_children')->willReturn($pages);
        $category->expects($this->any())->method('check_access')->willReturn($can_access);
        $category->expects($this->any())->method('is_hidden')->willReturn($hidden);
        return $category;
    }

    private function mock_admin_settings_page(string $name = 'test', string $visiblename = 'Test', $can_access = true, $hidden = false) {
        $page = $this->getMockBuilder(\admin_settingpage::class)
            ->onlyMethods(['check_access', 'is_hidden'])
            ->setConstructorArgs([$name, $visiblename, 'moodle/site:config', $hidden, null])
            ->getMock();
        $page->expects($this->any())->method('check_access')->willReturn($can_access);
        $page->expects($this->any())->method('is_hidden')->willReturn($hidden);
        return $page;
    }

    private static function get_items(map $map) {
        $property = new \ReflectionProperty(map::class, 'items');
        $property->setAccessible(true);
        return $property->getValue($map);
    }

    private static function get_groups(map $map) {
        $property = new \ReflectionProperty(map::class, 'groups');
        $property->setAccessible(true);
        return $property->getValue($map);
    }

    private static function get_keymap(map $map) {
        $property = new \ReflectionProperty(map::class, 'keymap');
        $property->setAccessible(true);
        return $property->getValue($map);
    }

    private static function assert_item_equals(string $key, string $label, ?string $url, item $item) {
        self::assertSame($key, $item->get_key(), 'The key for the given item is not as expected.');
        self::assertSame($label, (string)$item->get_label(), 'The label for the given item is no as expected.');
        if ($url !== null) {
            self::assertSame($url, (string)$item->get_url()->out_as_local_url(false));
        }
    }

    public function test_add_group__non_existent_admin_page() {
        $map = new map();
        $this->mock_admin_root($map);

        $map->add_group(new group([
            new item('test', 'thanks', 'core'),
        ]));

        self::assertEmpty(self::get_keymap($map));
        self::assertEmpty(self::get_items($map));
        $groups = self::get_groups($map);
        self::assertCount(1, $groups);
        $group = $groups[0];
        self::assertInstanceOf(group::class, $group);
        /** @var group $group */
        self::assertCount(1, $group->get_items());
        $item = $group->get_items()[0];
        self::assertInstanceOf(item::class, $item);
        /** @var item $item */
        self::assert_item_equals('test', 'Thanks', null, $item);
    }

    public function test_add_group__single_group_single_valid_adminpage() {
        $map = new map();
        $root = $this->mock_admin_root($map);
        $page = $this->mock_admin_externalpage();
        $root->method('locate')->willReturn($page);
        $map->add_group(new group([
            new item('test', 'thanks', 'core'),
        ]));

        self::assertEmpty(self::get_keymap($map));
        self::assertEmpty(self::get_items($map));
        $groups = self::get_groups($map);
        self::assertCount(1, $groups);
        $group = $groups[0];
        self::assertInstanceOf(group::class, $group);
        /** @var group $group */
        self::assertCount(1, $group->get_items());
        $item = $group->get_items()[0];
        self::assertInstanceOf(item::class, $item);
        /** @var item $item */
        self::assert_item_equals('test', 'Thanks', null, $item);
    }

    public function test_add_group__single_group_multiple_valid_adminpage() {
        $map = new map();
        $root = $this->mock_admin_root($map);

        $root->expects($this->any())->method('locate')->will(
            $this->returnValueMap(
                [
                    ['a', $this->mock_admin_externalpage('a', 'A', '/a')],
                    ['b', $this->mock_admin_externalpage('b', 'B', '/b')],
                    ['c', $this->mock_admin_externalpage('c', 'C', '/c')],
                ]
            )
        );

        $group = new group([
            new item('b', 'sort', 'core'),
            new item('c', 'from', 'core'),
            new item('a', 'thanks', 'core'),
        ]);
        $map->add_group($group);

        self::assertEmpty(self::get_keymap($map));
        self::assertEmpty(self::get_items($map));
        $groups = self::get_groups($map);
        self::assertCount(1, $groups);
        $group = $groups[0];
        self::assertInstanceOf(group::class, $group);
        /** @var group $group */
        self::assertCount(3, $group->get_items());
        self::assertInstanceOf(item::class, $group->get_items()[0]);
        self::assert_item_equals('b', 'Sort', null, $group->get_items()[0]);
        self::assertInstanceOf(item::class, $group->get_items()[1]);
        self::assert_item_equals('c', 'From', null, $group->get_items()[1]);
        self::assertInstanceOf(item::class, $group->get_items()[2]);
        self::assert_item_equals('a', 'Thanks', null, $group->get_items()[2]);
    }

    public function test_add_group__multiple_group_multiple_valid_adminpage() {
        $map = new map();
        $root = $this->mock_admin_root($map);

        $root->expects($this->any())->method('locate')->will(
            $this->returnValueMap(
                [
                    ['a', $this->mock_admin_externalpage('a', 'A', '/a')],
                    ['b', $this->mock_admin_externalpage('b', 'B', '/b')],
                    ['c', $this->mock_admin_externalpage('c', 'C', '/c')],
                ]
            )
        );

        $group1 = new group([
            new item('b', 'sort', 'core'),
            new item('a', 'thanks', 'core'),
        ]);
        $map->add_group($group1);
        $group2 = new group([
            new item('c', 'from', 'core'),
        ]);
        $map->add_group($group2);


        self::assertEmpty(self::get_keymap($map));
        self::assertEmpty(self::get_items($map));
        $groups = self::get_groups($map);
        self::assertCount(2, $groups);
        self::assertSame($group1, $groups[0]);
        self::assertSame($group2, $groups[1]);
    }

    public function test_add_group__multiple_item_some_valid_adminpage() {
        $map = new map();
        $root = $this->mock_admin_root($map);

        $root->expects($this->any())->method('locate')->will(
            $this->returnValueMap(
                [
                    ['a', $this->mock_admin_externalpage('a', 'A', '/a')],
                    ['c', $this->mock_admin_externalpage('c', 'C', '/c')],
                ]
            )
        );

        $group1 = new group([
            new item('b', 'sort', 'core'),
            new item('a', 'thanks', 'core'),
        ]);
        $map->add_group($group1);
        $group2 = new group([
            new item('c', 'from', 'core'),
        ]);
        $map->add_group($group2);


        self::assertEmpty(self::get_keymap($map));
        self::assertEmpty(self::get_items($map));
        $groups = self::get_groups($map);
        self::assertCount(2, $groups);
        self::assertSame($group1, $groups[0]);
        self::assertSame($group2, $groups[1]);
    }

    public function test_add_item__valid_item() {
        $map = new map();
        self::assertEmpty($this->get_items($map));
        $item = new item('test', 'thanks', 'core');
        $map->add_item($item);
        self::assertSame($item, $this->get_items($map)[0]);
    }

    public function test_get_mapped_items__several_valid_items() {
        $map = new map([
            new group([
                new item('b', 'sort', 'core'),
                new item('a', 'thanks', 'core'),
            ]),
            new group([
                new item('c', 'from', 'core'),
            ]),
        ]);
        $root = $this->mock_admin_root($map);

        $root->expects($this->any())->method('locate')->will(
            $this->returnValueMap(
                [
                    ['a', $this->mock_admin_externalpage('a', 'A', '/a')],
                    ['b', $this->mock_admin_externalpage('b', 'B', '/b')],
                    ['c', $this->mock_admin_externalpage('c', 'C', '/c')],
                ]
            )
        );

        $items = $map->get_mapped_items('a');

        self::assertCount(2, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('a', $items);
        self::assertArrayHasKey('b', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('b', 'Sort', '/b', $items['b']);
        self::assert_item_equals('a', 'Thanks', '/a', $items['a']);
        // Ensure they're in the right order.
        self::assertSame('Sort', (string)(array_shift($items))->get_label());
        self::assertSame('Thanks', (string)(array_shift($items))->get_label());

        $items = $map->get_mapped_items('b');

        self::assertCount(2, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('a', $items);
        self::assertArrayHasKey('b', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('b', 'Sort', '/b', $items['b']);
        self::assert_item_equals('a', 'Thanks', '/a', $items['a']);
        // Ensure they're in the right order.
        self::assertSame('Sort', (string)(array_shift($items))->get_label());
        self::assertSame('Thanks', (string)(array_shift($items))->get_label());

        $items = $map->get_mapped_items('c');

        self::assertCount(1, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('c', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('c', 'From', '/c', $items['c']);
    }

    public function test_prepare_to_cache__with_nothing_() {
        $map = new map();
        $root = $this->mock_admin_root($map);

        $expected = [
            'map' => [
            ],
            'items' => [
            ],
        ];
        self::assertEquals($expected, $map->prepare_to_cache());
    }

    public function test_prepare_to_cache__with_groups() {
        $map = new map([
            new group([
                new item('b', 'sort', 'core'),
                new item('a', 'thanks', 'core'),
            ]),
            new group([
                new item('c', 'from', 'core'),
            ]),
        ]);
        $root = $this->mock_admin_root($map);

        $root->expects($this->any())->method('locate')->will(
            $this->returnValueMap(
                [
                    ['a', $this->mock_admin_externalpage('a', 'A', '/a')],
                    ['b', $this->mock_admin_externalpage('b', 'B', '/b')],
                    ['c', $this->mock_admin_externalpage('c', 'C', '/c')],
                ]
            )
        );

        $expected = [
            'map' => [
                'b' => [
                    'b',
                    'a',
                ],
                'a' => [
                    'b',
                    'a',
                ],
                'c' => [
                    'c',
                ]
            ],
            'items' => [
                'b' => [
                    'b',
                    'sort',
                    'core',
                    '/b',
                    [
                        'b',
                        'a',
                    ],
                    []
                ],
                'a' => [
                    'a',
                    'thanks',
                    'core',
                    '/a',
                    [
                        'b',
                        'a',
                    ],
                    []
                ],
                'c' => [
                    'c',
                    'from',
                    'core',
                    '/c',
                    [
                        'c',
                    ],
                    []
                ]
            ],
        ];
        self::assertEquals($expected, $map->prepare_to_cache());
    }

    public function test_wake_from_cache() {
        $data = [
            'map' => [
                'b' => [
                    'b',
                    'a',
                ],
                'a' => [
                    'b',
                    'a',
                ],
                'c' => [
                    'c',
                ]
            ],
            'items' => [
                'b' => [
                    'b',
                    'sort',
                    'core',
                    '/b',
                    [
                        'b',
                        'a',
                    ],
                    []
                ],
                'a' => [
                    'a',
                    'thanks',
                    'core',
                    '/a',
                    [
                        'b',
                        'a',
                    ],
                    []
                ],
                'c' => [
                    'c',
                    'from',
                    'core',
                    '/c',
                    [
                        'c',
                    ],
                    []
                ]
            ],
        ];
        $map = map::wake_from_cache($data);
        $items = $map->get_mapped_items('a');

        self::assertCount(2, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('a', $items);
        self::assertArrayHasKey('b', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('b', 'Sort', '/b', $items['b']);
        self::assert_item_equals('a', 'Thanks', '/a', $items['a']);
        // Ensure they're in the right order.
        self::assertSame('Sort', (string)(array_shift($items))->get_label());
        self::assertSame('Thanks', (string)(array_shift($items))->get_label());

        $items = $map->get_mapped_items('b');

        self::assertCount(2, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('a', $items);
        self::assertArrayHasKey('b', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('b', 'Sort', '/b', $items['b']);
        self::assert_item_equals('a', 'Thanks', '/a', $items['a']);
        // Ensure they're in the right order.
        self::assertSame('Sort', (string)(array_shift($items))->get_label());
        self::assertSame('Thanks', (string)(array_shift($items))->get_label());

        $items = $map->get_mapped_items('c');

        self::assertCount(1, $items);
        // Ensure we've got the right keys.
        self::assertArrayHasKey('c', $items);
        // Ensure we've got the right items.
        self::assert_item_equals('c', 'From', '/c', $items['c']);

        self::assertSame($data, $map->prepare_to_cache());
    }

    public function test_construct_complex() {

        $groups = [
            new group([
                new item('b', 'sort', 'core'),
                new item('a', 'thanks', 'core'),
            ]),
            new group([
                new item('b', 'sort', 'core'),
                new item('c', 'plugginname', 'block_admin_related_pages'),
            ]),
            new group([
                new item('c', 'plugginname', 'block_admin_related_pages'),
                new item('d', 'plugginname', 'block_admin_related_pages'),
                new item('e', 'plugginname', 'block_admin_related_pages'),
            ]),
            new group([
                new item('g', 'plugginname', 'block_admin_related_pages', ['x']),
                new item('h', 'plugginname', 'block_admin_related_pages', ['y']),
            ]),
        ];
        $items = [
            new item('e', 'plugginname', 'block_admin_related_pages', ['a']),
            new item('e', 'plugginname', 'block_admin_related_pages', ['x']),
        ];

        $page_a = $this->mock_admin_externalpage('a', 'A', '/a');
        $page_b = $this->mock_admin_externalpage('b', 'B', '/b');
        $page_c = $this->mock_admin_externalpage('c', 'C', '/c');
        $page_d = $this->mock_admin_externalpage('d', 'D', '/d');
        $page_e = $this->mock_admin_externalpage('e', 'E', '/e');
        $page_f = $this->mock_admin_externalpage('f', 'F', '/f');
        $page_g = $this->mock_admin_externalpage('g', 'G', '/g');
        $page_h = $this->mock_admin_externalpage('h', 'H', '/h');
        $category_x = $this->mock_admin_category('x', 'X', [$page_g, $page_h]);
        $category_y = $this->mock_admin_category('y', 'Y', [$page_g, $page_f]);

        $map = new map($groups, $items);
        $root = $this->mock_admin_root($map);
        $root->expects($this->any())->method('locate')->will($this->returnValueMap([
            ['a', $page_a],
            ['b', $page_b],
            ['c', $page_c],
            ['d', $page_d],
            ['e', $page_e],
            ['f', $page_f],
            ['g', $page_g],
            ['h', $page_h],
            ['x', $category_x],
            ['y', $category_y],
        ]));

        $items = $map->get_mapped_items('a');
        self::assertCount(3, $items);
        self::assertSame(['b', 'a', 'e'], array_keys($items));

        $items = $map->get_mapped_items('b');
        self::assertCount(3, $items);
        self::assertSame(['b', 'a', 'c'], array_keys($items));

        $items = $map->get_mapped_items('c');
        self::assertCount(4, $items);
        self::assertSame(['b', 'c', 'd', 'e'], array_keys($items));

        $items = $map->get_mapped_items('d');
        self::assertCount(3, $items);
        self::assertSame(['c', 'd', 'e'], array_keys($items));

        $items = $map->get_mapped_items('e');
        self::assertCount(3, $items);
        self::assertSame(['c', 'd', 'e'], array_keys($items));

        $items = $map->get_mapped_items('f');
        self::assertCount(2, $items);
        self::assertSame(['g', 'h'], array_keys($items));

        $items = $map->get_mapped_items('g');
        self::assertCount(3, $items);
        self::assertSame(['g', 'h', 'e'], array_keys($items));

        $items = $map->get_mapped_items('h');
        self::assertCount(3, $items);
        self::assertSame(['g', 'h', 'e'], array_keys($items));

    }

}
