<?php

// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * IP Lookup utility functions
 *
 * @package    core
 * @subpackage iplookup
 * @copyright  2010 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

/**
 * Returns location information
 * @param string $ip
 * @return array
 */
function iplookup_find_location($ip) {
    global $CFG;

    $info = array('city'=>null, 'country'=>null, 'longitude'=>null, 'latitude'=>null, 'error'=>null, 'note'=>'',  'title'=>array());

    if (!empty($CFG->geoip2file) and file_exists($CFG->geoip2file)) {
        $reader = new GeoIp2\Database\Reader($CFG->geoip2file);
        $record = $reader->city($ip);

        if (empty($record)) {
            $info['error'] = get_string('iplookupfailed', 'error', $ip);
            return $info;
        }

        $info['city'] = core_text::convert($record->city->name, 'iso-8859-1', 'utf-8');
        $info['title'][] = $info['city'];

        $countrycode = $record->country->isoCode;
        $countries = get_string_manager()->get_list_of_countries(true);
        if (isset($countries[$countrycode])) {
            // Prefer our localized country names.
            $info['country'] = $countries[$countrycode];
        } else {
            $info['country'] = $record->country->names['en'];
        }
        $info['title'][] = $info['country'];

        $info['longitude'] = $record->location->longitude;
        $info['latitude']  = $record->location->latitude;
        $info['note'] = get_string('iplookupmaxmindnote', 'admin');

        return $info;

    }

    return ['error' => get_string('iplookupnolocal', 'admin')];
}

/**
 * @param int|null $user_id
 * @param int|null $target_user_id
 * @return bool
 */
function ip_lookup_enabled_for_user(?int $user_id = null, ?int $target_user_id = null, bool $skip_capability_check = false): bool {
    global $CFG, $USER;
    static $available = [];

    $user_id ??= $USER->id;
    $key = $user_id . ':' . ($target_user_id ?? '0');

    // In a request, only check each user *once*
    // This is needed if this report is ever exported as part of a CLI task
    if (isset($available[$key])) {
        return $available[$key];
    }

    // Enabled at all?
    if (empty($CFG->geoip2file) || !file_exists($CFG->geoip2file)) {
        $available[$key] = false;
        return false;
    }

    // Capability check can be opted out until Totara 20
    if (isset($CFG->revert_TL_41079_until_T20) && $CFG->revert_TL_41079_until_T20) {
        $available[$key] = true;
        return true;
    }

    if ($skip_capability_check) {
        $available[$key] = true;
        return true;
    }

    // If it's for a target user id, then you need the capability to view their userid
    if ($target_user_id !== null) {
        $user_context = context_user::instance($target_user_id);
        if (has_capability('moodle/user:viewlastip', $user_context)) {
            $available[$key] = true;
            return true;
        }
    }

    // Otherwise search the system context instead for general report view
    if (has_capability('report/log:view', context_system::instance(), $user_id)) {
        $available[$key] = true;
        return true;
    }

    $available[$key] = false;
    return false;
}