<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * GeoIP tests
 *
 * @package    core_iplookup
 * @category   phpunit
 * @copyright  2012 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

/**
 * GeoIp data file parsing test.
 */
class core_iplookup_geoip_test extends \core_phpunit\testcase {
    public function setUp(): void {
        global $CFG;
        require_once("{$CFG->libdir}/filelib.php");
        require_once("{$CFG->dirroot}/iplookup/lib.php");
    }

    /**
     * Test the format of data returned in the iplookup_find_location function.
     *
     * @dataProvider ip_provider
     * @param   string  $ip The IP to test
     */
    public function test_ip($ip) {
        global $CFG;

        if (empty($CFG->geoip2file) || !file_exists($CFG->geoip2file)) {
            $this->markTestSkipped('GeoIP test disabled, no copy of the database held locally.');
        }

        // Note: The results we get from the iplookup tests are beyond our control.
        // We used to check a specific IP to a known location, but these have become less reliable and change too
        // frequently to be used for testing.

        $result = iplookup_find_location($ip);

        $this->assertIsArray($result);
        $this->assertIsFloat($result['latitude']);
        $this->assertIsFloat($result['longitude']);
        $this->assertIsString($result['city']);
        $this->assertIsString($result['country']);
        $this->assertIsArray($result['title']);
        $this->assertIsString($result['title'][0]);
        $this->assertIsString($result['title'][1]);
        $this->assertNull($result['error']);
    }

    /**
     * Data provider for IP lookup test.
     *
     * @return array
     */
    public static function ip_provider() {
        return [
            'IPv4: Sample suggested by maxmind themselves' => ['24.24.24.24'],
            'IPv4: github.com' =>  ['192.30.255.112'],
            'IPv6: UCLA' => ['2607:f010:3fe:fff1::ff:fe00:25'],
        ];
    }

    /**
     * @return void
     */
    public function test_ip_lookup_enabled_for_user(): void {
        global $CFG;

        // Scenario: Feature isn't enabled
        $CFG->geoip2file = null;
        $user = $this->getDataGenerator()->create_user();
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertFalse($result);

        // Scenario: Cached result returned from above
        $CFG->geoip2file = __DIR__ . '/fixtures/dummy_geoip_file.txt';
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertFalse($result);

        // Scenario: T20 revert is enabled
        $CFG->revert_TL_41079_until_T20 = true;
        $user = $this->getDataGenerator()->create_user();
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertTrue($result);

        // Scenario: T20 revert is disabled - capability check occurs
        $CFG->revert_TL_41079_until_T20 = false;
        $user = $this->getDataGenerator()->create_user();
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertFalse($result);

        // Scenario: User has the report view capability in the system context
        $user = $this->getDataGenerator()->create_user();
        $role_id = $this->getDataGenerator()->create_role();
        assign_capability('report/log:view', CAP_ALLOW, $role_id, context_system::instance());
        $this->getDataGenerator()->role_assign($role_id, $user->id);
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertTrue($result);

        // Scenario: User caching kicks in once again
        unassign_capability('report/log:view', $role_id, context_system::instance());
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertTrue($result);

        $user = $this->getDataGenerator()->create_user();
        $result = ip_lookup_enabled_for_user($user->id);
        $this->assertFalse($result);
    }
}
