<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2019 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Ning Zhou <ning.zhou@totara.com>
 * @package core
 */

namespace core\entity;

use core\orm\entity\entity;
use core\orm\entity\relations\has_many;
use core\orm\entity\relations\has_one;
use core\orm\entity\repository;

/**
 * Course categories entity
 *
 * @property-read int $id ID
 * @property string $name
 * @property string $idnumber
 * @property string $description
 * @property int $descriptionformat
 * @property int $parent
 * @property int $sortorder
 * @property int $coursecount
 * @property int $visible
 * @property int $visibleold
 * @property int $timemodified
 * @property int $depth
 * @property string $path
 * @property string $theme
 * @property int $programcount
 * @property int $certifcount
 * @property int $issystem
 * @property int $iscontainer
 *
 * @property-read course_categories[]| $children
 *
 * Relationships:
 * @property-read course[] $courses
 * @property-read tenant $tenant
 *
 * @package criteria_coursecompletion/entity
 */
class course_categories extends entity {
    public const TABLE = 'course_categories';

    /**
     * Get courses in this category
     *
     * @return has_many
     */
    public function courses(): has_many {
        return $this->has_many(course::class, 'category');
    }

    /**
     * Get tenant by category id
     *
     * @return has_one
     */
    public function tenant(): has_one {
        return $this->has_one(tenant::class, 'categoryid');
    }

    /**
     * Get base query of getting children
     *
     * @return repository
     * @throws \coding_exception
     */
    private function get_children_base(): repository {
        return static::repository()
            ->where('parent', '=', $this->id)
            ->order_by('sortorder', 'ASC')
            ->order_by('id', 'ASC');
    }

    /**
     * Get all child categories for current category
     *
     * @return course_categories[] | []
     */
    public function get_children_attribute(): array {
        return $this->get_children_base()
            ->get()
            ->all();
    }
}