<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.


/**
 * Core external functions and service definitions.
 *
 * The functions and services defined on this file are
 * processed and registered into the Moodle DB after any
 * install or upgrade operation. All plugins support this.
 *
 * For more information, take a look to the documentation available:
 *     - Webservices API: {@link http://docs.moodle.org/dev/Web_services_API}
 *     - External API: {@link http://docs.moodle.org/dev/External_functions_API}
 *     - Upgrade API: {@link http://docs.moodle.org/dev/Upgrade_API}
 *
 * @package    core_webservice
 * @category   webservice
 * @copyright  2009 Petr Skodak
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

$functions = array(
    'core_badges_get_user_badges' => array(
        'classname'     => 'core_badges_external',
        'methodname'    => 'get_user_badges',
        'description'   => 'Returns the list of badges awarded to a user.',
        'type'          => 'read',
        'capabilities'  => 'moodle/badges:viewotherbadges',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_create_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'create_calendar_events',
        'description' => 'Create calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries'
    ),
    'core_calendar_delete_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'delete_calendar_events',
        'description' => 'Delete calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries'
    ),
    'core_calendar_get_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_events',
        'description' => 'Get calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_cohort_add_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'add_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Adds cohort members.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:assign'
    ),
    'core_cohort_create_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'create_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Creates new cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_cohort_delete_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'delete_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Deletes cohort members.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:assign'
    ),
    'core_cohort_delete_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'delete_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Deletes all specified cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_cohort_get_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'get_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Returns cohort members.',
        'type' => 'read',
        'capabilities' => 'moodle/cohort:view'
    ),
    'core_cohort_get_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'get_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Returns cohort details.',
        'type' => 'read',
        'capabilities' => 'moodle/cohort:view'
    ),
    'core_cohort_update_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'update_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Updates existing cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_comment_get_comments' => array(
        'classname' => 'core_comment_external',
        'methodname' => 'get_comments',
        'description' => 'Returns comments.',
        'type' => 'read',
        'capabilities' => 'moodle/comment:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_get_activities_completion_status' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'get_activities_completion_status',
        'description' => 'Return the activities completion status for a user in a course.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_get_course_completion_status' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'get_course_completion_status',
        'description' => 'Returns course completion status.',
        'type' => 'read',
        'capabilities' => 'report/completion:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_mark_course_self_completed' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'mark_course_self_completed',
        'description' => 'Update the course completion status for the current user (if course self-completion is enabled).',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_update_activity_completion_status_manually' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'update_activity_completion_status_manually',
        'description' => 'Update completion status for the current user in an activity, only for activities with manual tracking.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_create_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'create_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Create course categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage'
    ),
    'core_course_create_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'create_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Create new courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:create, moodle/course:visibility'
    ),
    'core_course_delete_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Delete course categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage'
    ),
    'core_course_delete_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Deletes all specified courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:delete'
    ),
    'core_course_delete_modules' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_modules',
        'classpath' => 'course/externallib.php',
        'description' => 'Deletes all specified module instances',
        'type' => 'write',
        'capabilities' => 'moodle/course:manageactivities'
    ),
    'core_course_duplicate_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'duplicate_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Duplicate an existing course (creating a new one).',
        'type' => 'write',
        'capabilities' => 'moodle/backup:backupcourse, moodle/restore:restorecourse, moodle/course:create'
    ),
    'core_course_get_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Return category details',
        'type' => 'read',
        'capabilities' => 'moodle/category:viewhiddencategories',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_contents' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_contents',
        'classpath' => 'course/externallib.php',
        'description' => 'Get course contents',
        'type' => 'read',
        'capabilities' => 'moodle/course:update, moodle/course:viewhiddencourses',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_course_module' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_module',
        'classpath' => 'course/externallib.php',
        'description' => 'Return information about a course module',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_course_module_by_instance' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_module_by_instance',
        'classpath' => 'course/externallib.php',
        'description' => 'Return information about a given module name and instance id',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Return course details',
        'type' => 'read',
        'capabilities' => 'moodle/course:view, moodle/course:update, moodle/course:viewhiddencourses',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_import_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'import_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Import course data from a course into another course. Does not include any user data.',
        'type' => 'write',
        'capabilities' => 'moodle/backup:backuptargetimport, moodle/restore:restoretargetimport'
    ),
    'core_course_search_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'search_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Search courses by (name, module, block, tag)',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_update_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'update_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Update categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage',
    ),
    'core_course_update_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'update_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Update courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:update, moodle/course:changecategory, moodle/course:changefullname, '
            . 'moodle/course:changeshortname, moodle/course:changeidnumber, moodle/course:changesummary, moodle/course:visibility'
    ),
    'core_course_view_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'view_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Log that the course was viewed',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_activities_overview' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_activities_overview',
        'classpath' => 'course/externallib.php',
        'description' => 'Return activities overview for the given courses.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_user_navigation_options' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_user_navigation_options',
        'classpath' => 'course/externallib.php',
        'description' => 'Return a list of navigation options in a set of courses that are avaialable or not for the current user.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_user_administration_options' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_user_administration_options',
        'classpath' => 'course/externallib.php',
        'description' => 'Return a list of administration options in a set of courses that are avaialable or not for the current
                            user.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_courses_by_field' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_courses_by_field',
        'classpath' => 'course/externallib.php',
        'description' => 'Get courses matching a specific field (id/s, shortname, idnumber, category)',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_check_updates' => array(
        'classname' => 'core_course_external',
        'methodname' => 'check_updates',
        'classpath' => 'course/externallib.php',
        'description' => 'Check if there is updates affecting the user for the given course and contexts.',
        'type' => 'read',
        // Totara: no ajax
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_updates_since' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_updates_since',
        'classpath' => 'course/externallib.php',
        'description' => 'Check if there are updates affecting the user for the given course since the given time stamp.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_get_course_enrolment_methods' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_course_enrolment_methods',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get the list of course enrolment methods',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_get_enrolled_users' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_enrolled_users',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get enrolled users by course id.',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update, '
            . 'moodle/site:accessallgroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_get_enrolled_users_with_capability' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_enrolled_users_with_capability',
        'classpath' => 'enrol/externallib.php',
        'description' => 'For each course and capability specified, return a list of the users that are enrolled in the course
                                  and have that capability',
        'type' => 'read',
    ),
    'core_enrol_get_users_courses' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_users_courses',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get the list of courses where a user is enrolled in',
        'type' => 'read',
        'capabilities' => 'moodle/course:viewparticipants',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_fetch_notifications' => array(
        'classname' => 'core_external',
        'methodname' => 'fetch_notifications',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a list of notifications for the current session',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_files_get_files' => array(
        'classname' => 'core_files_external',
        'methodname' => 'get_files',
        'description' => 'browse moodle files',
        'type' => 'read',
        'classpath' => 'files/externallib.php',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_files_upload' => array(
        'classname' => 'core_files_external',
        'methodname' => 'upload',
        'description' => 'upload a file to moodle',
        'type'        => 'write',
        'classpath'   => 'files/externallib.php',
    ),
    'core_form_get_filetypes_browser_data' => array(
        'classname' => 'core_form\external',
        'methodname' => 'get_filetypes_browser_data',
        'classpath' => '',
        'description' => 'Provides data for the filetypes element browser.',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_get_component_strings' => array(
        'classname' => 'core_external',
        'methodname' => 'get_component_strings',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return all raw strings (with {$a->xxx}), for a specific component ' .
            '- similar to core get_component_strings(), call',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_get_fragment' => array(
        'classname' => 'core_external',
        'methodname' => 'get_fragment',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a fragment for inclusion, such as a JavaScript page.',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_get_string' => array(
        'classname' => 'core_external',
        'methodname' => 'get_string',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a translated string - similar to core get_string(), call',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_get_strings' => array(
        'classname' => 'core_external',
        'methodname' => 'get_strings',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return some translated strings - like several core get_string(), calls',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_get_user_dates' => array(
        'classname' => 'core_external',
        'methodname' => 'get_user_dates',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return formatted timestamps',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_grades_update_grades' => array(
        'classname' => 'core_grades_external',
        'methodname' => 'update_grades',
        'description' => 'Update a grade item and associated student grades.',
        'type' => 'write',
    ),
    'core_grading_get_definitions' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'get_definitions',
        'description' => 'Get grading definitions',
        'type' => 'read',
    ),
    'core_grading_get_gradingform_instances' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'get_gradingform_instances',
        'description' => 'Get grading form instances',
        'type' => 'read',
    ),
    'core_grading_save_definitions' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'save_definitions',
        'description' => 'Save grading definitions',
        'type' => 'write',
    ),
    'core_group_add_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'add_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Adds group members.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups',
    ),
    'core_group_assign_grouping' => array(
        'classname' => 'core_group_external',
        'methodname' => 'assign_grouping',
        'classpath' => 'group/externallib.php',
        'description' => 'Assing groups from groupings',
        'type' => 'write',
    ),
    'core_group_create_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'create_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Creates new groupings',
        'type' => 'write',
    ),
    'core_group_create_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'create_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Creates new groups.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_delete_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes group members.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_delete_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes all specified groupings.',
        'type' => 'write',
    ),
    'core_group_delete_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes all specified groups.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_get_activity_allowed_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_activity_allowed_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Gets a list of groups that the user is allowed to access within the specified activity.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_activity_groupmode' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_activity_groupmode',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns effective groupmode used in a given activity.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_course_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groupings in specified course.',
        'type' => 'read',
    ),
    'core_group_get_course_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groups in specified course.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_get_course_user_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_user_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groups in specified course for the specified user.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns group members.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_get_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns groupings details.',
        'type' => 'read',
    ),
    'core_group_get_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns group details.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_unassign_grouping' => array(
        'classname' => 'core_group_external',
        'methodname' => 'unassign_grouping',
        'classpath' => 'group/externallib.php',
        'description' => 'Unassing groups from groupings',
        'type' => 'write',
    ),
    'core_group_update_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'update_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Updates existing groupings',
        'type' => 'write',
    ),
    'core_message_block_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'block_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Block contacts',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_create_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'create_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Add contacts to the contact list',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Remove contacts from the contact list',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_conversation' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_conversation',
        'classpath' => 'message/externallib.php',
        'description' => 'Deletes a conversation.',
        'type' => 'write',
        'capabilities' => 'moodle/site:deleteownmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_message' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_message',
        'classpath' => 'message/externallib.php',
        'description' => 'Deletes a message.',
        'type' => 'write',
        'capabilities' => 'moodle/site:deleteownmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_blocked_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_blocked_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of users blocked',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_data_for_messagearea_search_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_search_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for searching for messages',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_data_for_messagearea_search_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_search_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for searching for people',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_search_users_in_course' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_search_users_in_course',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for searching for people in a course',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for the conversation list',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for the contact list',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for the messages',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_get_most_recent_message' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_get_most_recent_message',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for the most recent message',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_data_for_messagearea_get_profile' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_get_profile',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for the users\'s profile',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_get_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the contact list',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of messages sent and received by a user (conversations, notifications or both)',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_unread_conversations_count' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_unread_conversations_count',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the count of unread conversations for a given user',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_mark_all_notifications_as_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_all_notifications_as_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark all notifications as read for a given user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_mark_all_messages_as_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_all_messages_as_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark all messages as read for a given user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_mark_message_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_message_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark a single message as read, trigger message_viewed event.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_mark_notification_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_notification_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark a single notification as read, trigger notification_viewed event.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_message_processor_config_form' => array(
        'classname' => 'core_message_external',
        'methodname' => 'message_processor_config_form',
        'classpath' => 'message/externallib.php',
        'description' => 'Process the message processor config form',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_message_processor' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_message_processor',
        'classpath' => 'message/externallib.php',
        'description' => 'Get a message processor',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_search_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'search_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Search for contacts',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_send_instant_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'send_instant_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Send instant messages',
        'type' => 'write',
        'capabilities' => 'moodle/site:sendmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_unblock_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'unblock_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Unblock contacts',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_user_notification_preferences' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_user_notification_preferences',
        'classpath' => 'message/externallib.php',
        'description' => 'Get the notification preferences for a given user.',
        'type' => 'read',
        'capabilities' => 'moodle/user:editownmessageprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_user_message_preferences' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_user_message_preferences',
        'classpath' => 'message/externallib.php',
        'description' => 'Get the message preferences for a given user.',
        'type' => 'read',
        'capabilities' => 'moodle/user:editownmessageprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_create_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'create_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Create notes',
        'type' => 'write',
        'capabilities' => 'moodle/notes:manage',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_delete_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'delete_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Delete notes',
        'type' => 'write',
        'capabilities' => 'moodle/notes:manage',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_get_course_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'get_course_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Returns all notes in specified course (or site), for the specified user.',
        'type' => 'read',
        'capabilities' => 'moodle/notes:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_get_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'get_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Get notes',
        'type' => 'read',
        'capabilities' => 'moodle/notes:view'
    ),
    'core_notes_update_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'update_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Update notes',
        'type' => 'write',
        'capabilities' => 'moodle/notes:manage'
    ),
    'core_notes_view_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'view_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Simulates the web interface view of notes/index.php: trigger events.',
        'type' => 'write',
        'capabilities' => 'moodle/notes:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_output_load_template' => array(
        'classname' => 'core\output\external',
        'methodname' => 'load_template',
        'description' => 'Load a template for a renderable',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    // Question related functions.
    'core_question_update_flag' => array(
        'classname'     => 'core_question_external',
        'methodname'    => 'update_flag',
        'description'   => 'Update the flag state of a question attempt.',
        'type'          => 'write',
        'capabilities'  => 'moodle/question:flag',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_rating_get_item_ratings' => array(
        'classname' => 'core_rating_external',
        'methodname' => 'get_item_ratings',
        'description' => 'Retrieve all the ratings for an item.',
        'type' => 'read',
        'capabilities' => 'moodle/rating:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_rating_add_rating' => array(
        'classname' => 'core_rating_external',
        'methodname' => 'add_rating',
        'description' => 'Rates an item.',
        'type' => 'write',
        'capabilities' => 'moodle/rating:rate',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_role_assign_roles' => array(
        'classname' => 'core_role_external',
        'methodname' => 'assign_roles',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Manual role assignments.',
        'type' => 'write',
        'capabilities' => 'moodle/role:assign'
    ),
    'core_role_unassign_roles' => array(
        'classname' => 'core_role_external',
        'methodname' => 'unassign_roles',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Manual role unassignments.',
        'type' => 'write',
        'capabilities' => 'moodle/role:assign'
    ),
    'core_tag_get_tagindex' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tagindex',
        'description' => 'Gets tag index page for one tag and one tag area',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_tag_get_tags' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tags',
        'description' => 'Gets tags by their ids',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_tag_update_tags' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'update_tags',
        'description' => 'Updates tags',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_update_inplace_editable' => array(
        'classname' => 'core_external',
        'methodname' => 'update_inplace_editable',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Generic service to update title',
        'type' => 'write',
        'loginrequired' => true,
        'ajax' => true,
    ),
    'core_user_add_user_device' => array(
        'classname' => 'core_user_external',
        'methodname' => 'add_user_device',
        'classpath' => 'user/externallib.php',
        'description' => 'Store mobile user devices information for PUSH Notifications.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // TOTARA: support for getting flex icons list.
    'core_output_get_flex_icons' => array(
        'classname'   => 'core\output\external',
        'methodname'  => 'get_flex_icons',
        'description' => 'Get list of flex icon definitions for the specified theme',
        'type'        => 'read',
        'loginrequired' => false,
        'ajax'        => true,
    ),

    'core_user_add_user_private_files' => array(
        'classname' => 'core_user_external',
        'methodname' => 'add_user_private_files',
        'classpath' => 'user/externallib.php',
        'description' => 'Copy files from a draft area to users private files area.',
        'type' => 'write',
        'capabilities' => 'moodle/user:manageownfiles',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_create_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'create_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Create users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:create'
    ),
    'core_user_delete_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'delete_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Delete users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:delete'
    ),
    'core_user_get_course_user_profiles' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_course_user_profiles',
        'classpath' => 'user/externallib.php',
        'description' => 'Get course user profiles (each of the profils matching a course id and a user id),.',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update, '
            . 'moodle/site:accessallgroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_get_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_users',
        'classpath' => 'user/externallib.php',
        'description' => 'search for users matching the parameters',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update'
    ),
    'core_user_get_users_by_field' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_users_by_field',
        'classpath' => 'user/externallib.php',
        'description' => 'Retrieve users\' information for a specified unique field - If you want to do a user search, use '
            . 'core_user_get_users()',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_remove_user_device' => array(
        'classname' => 'core_user_external',
        'methodname' => 'remove_user_device',
        'classpath' => 'user/externallib.php',
        'description' => 'Remove a user device from the Moodle database.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_update_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Update users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:update',
        // Totara: no ajax!
    ),
    'core_user_update_user_preferences' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_user_preferences',
        'classpath' => 'user/externallib.php',
        'description' => 'Update a user\'s preferences',
        'type' => 'write',
        'capabilities' => 'moodle/user:editownmessageprofile, moodle/user:editmessageprofile',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_view_user_list' => array(
        'classname' => 'core_user_external',
        'methodname' => 'view_user_list',
        'classpath' => 'user/externallib.php',
        'description' => 'Simulates the web-interface view of user/index.php (triggering events),.',
        'type' => 'write',
        'capabilities' => 'moodle/course:viewparticipants',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_view_user_profile' => array(
        'classname' => 'core_user_external',
        'methodname' => 'view_user_profile',
        'classpath' => 'user/externallib.php',
        'description' => 'Simulates the web-interface view of user/profile.php (triggering events),.',
        'type' => 'write',
        'capabilities' => 'moodle/user:viewdetails',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_update_picture' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_picture',
        'classpath' => 'user/externallib.php',
        'description' => 'Update or delete the user picture in the site',
        'type' => 'write',
        'capabilities' => 'moodle/user:editownprofile, moodle/user:editprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_agree_site_policy' => array(
        'classname' => 'core_user_external',
        'methodname' => 'agree_site_policy',
        'classpath' => 'user/externallib.php',
        'description' => 'Agree the site policy for the current user.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // Totara: No competencies functions!

    'core_webservice_get_site_info' => array(
        'classname' => 'core_webservice_external',
        'methodname' => 'get_site_info',
        'classpath' => 'webservice/externallib.php',
        'description' => 'Return some site info / user info / list web service functions',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // Blocks functions.
    'core_block_get_course_blocks' => array(
        'classname'     => 'core_block_external',
        'methodname'    => 'get_course_blocks',
        'description'   => 'Returns blocks information for a course.',
        'type'          => 'read',
        'capabilities'  => '',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

);

$services = array(
    // Totara: no services by default.
);
