<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.


/**
 * Text type form element
 *
 * Contains HTML class for a text type element
 *
 * @package   core_form
 * @copyright 2006 Jamie Pratt <me@jamiep.org>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once("HTML/QuickForm/static.php");
require_once('templatable_form_element.php');

/**
 * Text type element
 *
 * HTML class for a text type element
 *
 * @package   core_form
 * @category  form
 * @copyright 2006 Jamie Pratt <me@jamiep.org>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class MoodleQuickForm_static extends HTML_QuickForm_static implements templatable {
    use templatable_form_element {
        export_for_template as export_for_template_base;
    }

    /** @var string Form element type */
    var $_elementTemplateType='static';

    /** @var string html for help button, if empty then no help */
    var $_helpbutton='';

    protected $allowxss = false;

    /**
     * constructor
     *
     * @param string $elementName (optional) name of the text field
     * @param string $elementLabel (optional) text field label
     * @param string $text (optional) Text to put in text field
     */
    public function __construct($elementName=null, $elementLabel=null, $text=null) {
        parent::__construct($elementName, $elementLabel, $text);
    }

    /**
     * Old syntax of class constructor. Deprecated in PHP7.
     *
     * @deprecated since Moodle 3.1
     */
    public function MoodleQuickForm_static($elementName=null, $elementLabel=null, $text=null) {
        debugging('Use of class name as constructor is deprecated', DEBUG_DEVELOPER);
        self::__construct($elementName, $elementLabel, $text);
    }

    /**
     * Set to true if static element uses unsafe or malformed HTML markup,
     * by default static HTML text is cleaned and normalised before output.
     *
     * NOTE: for backwards compatibility this is ignored if $CFG->disableconsistentcleaning is enabled
     *
     * @since Totara 13.0
     *
     * @param bool $value
     * @return MoodleQuickForm_static self instance to allow chaining
     */
    public function set_allow_xss(bool $value) {
        $this->allowxss = $value;
        return $this;
    }

    /**
     * get html for help button
     *
     * @return string html for help button
     */
    function getHelpButton(){
        return $this->_helpbutton;
    }

    /**
     * Gets the type of form element
     *
     * @return string
     */
    function getElementTemplateType(){
        return $this->_elementTemplateType;
    }

    public function toHtml() {
        global $CFG;

        // NOTE: do not use parent::toHtml()} here!

        $html = $this->_text;

        if (empty($CFG->disableconsistentcleaning) && !$this->allowxss) {
            $html = clean_text($html, FORMAT_HTML);
        }

        return $this->_getTabs() . $html;
    }

    public function export_for_template(renderer_base $output) {
        $context = $this->export_for_template_base($output);
        $context['html'] = $this->toHtml();
        $context['staticlabel'] = true;
        return $context;
    }
}
