<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Completion tests.
 *
 * @package    core_completion
 * @category   phpunit
 * @copyright  2008 Sam Marshall
 * @copyright  2013 Frédéric Massart
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use PHPUnit\Framework\MockObject\MockObject;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->libdir.'/completionlib.php');

class core_completionlib_test extends \core_phpunit\testcase {
    protected $course;
    protected $user;
    protected $module1;
    protected $module2;

    protected function tearDown(): void {
        $this->course = null;
        $this->user = null;
        $this->module1 = null;
        $this->module2 = null;
        parent::tearDown();
    }

    protected function mock_setup() {
        global $DB, $CFG, $USER;

        $DB = $this->createMock(get_class($DB));
        $CFG->enablecompletion = COMPLETION_ENABLED;
        $USER = (object)array('id' =>314159);
    }

    /**
     * Create course with user and activities.
     */
    protected function setup_data() {
        // Enable completion before creating modules, otherwise the completion data is not written in DB.
        set_config('enablecompletion', 1);

        // Create a course with activities.
        $this->course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $this->user = $this->getDataGenerator()->create_user();
        $this->getDataGenerator()->enrol_user($this->user->id, $this->course->id);

        $this->module1 = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id));
        $this->module2 = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id));
    }

    /**
     * Returns the core completion cache instance.
     *
     * @return cache_application
     */
    protected function get_completion_cache() {
        return cache::make('core', 'completion');
    }

    /**
     * Asserts that two variables are equal.
     *
     * @param  mixed   $expected
     * @param  mixed   $actual
     * @param  string  $message
     * @param  float   $delta
     * @param  integer $maxDepth
     * @param  boolean $canonicalize
     * @param  boolean $ignoreCase
     */
    public static function local_assert_equals($expected, $actual, string $message = '', float $delta = 0.0, int $maxDepth = 10, bool $canonicalize = false, bool $ignoreCase = false): void {
        if (is_object($expected) && is_object($actual)) {
            if (property_exists($expected, 'timemodified') && property_exists($actual, 'timemodified')) {
                if ($expected->timemodified + 1 == $actual->timemodified) {
                    $expected = clone($expected);
                    $expected->timemodified = $actual->timemodified;
                }
            }
        }
        parent::assertEquals($expected, $actual, $message);
    }

    public function test_is_enabled_for_site() {

        // Config alone.
        set_config('enablecompletion', 1);
        $this->assertTrue(completion_info::is_enabled_for_site());
        set_config('enablecompletion', 0);
        $this->assertFalse(completion_info::is_enabled_for_site());
    }

    public function test_is_enabled_for_course() {

        set_config('enablecompletion', 1);

        // Course.
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_DISABLED));
        $c = new completion_info($course);
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled());

        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));
        $c = new completion_info($course);
        $this->local_assert_equals(COMPLETION_ENABLED, $c->is_enabled());
        set_config('enablecompletion', 0);
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled());
    }

    public function test_is_enabled_for_module() {
        global $DB;

        set_config('enablecompletion', 1);
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));

        $completionnone = array('completion' => COMPLETION_TRACKING_NONE);
        $forum1 = $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionnone);
        $cm1 = get_coursemodule_from_instance('forum', $forum1->id);

        $completionmanual = array('completion' => COMPLETION_TRACKING_MANUAL);
        $forum2 = $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionmanual);
        $cm2 = get_coursemodule_from_instance('forum', $forum2->id);

        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $forum3 = $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionauto);
        $cm3 = get_coursemodule_from_instance('forum', $forum3->id);

        set_config('enablecompletion', 0);
        $c = new completion_info($course);
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm1));
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm2));
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm3));

        set_config('enablecompletion', 1);
        $c = new completion_info($course);
        $this->local_assert_equals(COMPLETION_TRACKING_NONE, $c->is_enabled($cm1));
        $this->local_assert_equals(COMPLETION_TRACKING_MANUAL, $c->is_enabled($cm2));
        $this->local_assert_equals(COMPLETION_TRACKING_AUTOMATIC, $c->is_enabled($cm3));

        $course->enablecompletion = (string)COMPLETION_DISABLED;
        $DB->update_record('course', $course);
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm1));
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm2));
        $this->local_assert_equals(COMPLETION_DISABLED, $c->is_enabled($cm3));
    }
    public function test_update_state(): void {
        $this->mock_setup();

        $mockbuilder = $this->getMockBuilder(completion_info::class);
        $mockbuilder->onlyMethods(array('is_enabled', 'get_data', 'internal_get_state', 'internal_set_data'));
        $mockbuilder->setConstructorArgs(array((object)array('id' => 42)));
        $cm = (object)array('id' => 13, 'course' => 42);

        // Not enabled, should do nothing.
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(false);
        $c->update_state($cm);

        // Enabled, but current state is same as possible result, do nothing.
        $cm->completion = COMPLETION_TRACKING_AUTOMATIC;
        $c = $mockbuilder->getMock();
        $current = (object)array('completionstate' => COMPLETION_COMPLETE, 'overrideby' => null);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->willReturn($current);
        $c->update_state($cm, COMPLETION_COMPLETE);

        // Enabled, but current state is a specific one and new state is just
        // complete, so do nothing.
        $c = $mockbuilder->getMock();
        $current->completionstate = COMPLETION_COMPLETE_PASS;
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->willReturn($current);
        $c->update_state($cm, COMPLETION_COMPLETE);

        // Manual, change state (no change).
        $c = $mockbuilder->getMock();
        $cm = (object)array('id' => 13, 'course' => 42, 'completion' => COMPLETION_TRACKING_MANUAL);
        $current->completionstate = COMPLETION_COMPLETE;
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->willReturn($current);
        $c->update_state($cm, COMPLETION_COMPLETE);

        // Manual, change state (change).
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->willReturn($current);
        $changed = clone($current);
        $changed->timemodified = time();
        $changed->completionstate = COMPLETION_INCOMPLETE;
        $c->expects($this->once())
            ->method('internal_set_data');
        $c->update_state($cm, COMPLETION_INCOMPLETE);

        // Auto, change state.
        $c = $mockbuilder->getMock();
        $cm = (object)array('id' => 13, 'course' => 42, 'module' => 1, 'completion' => COMPLETION_TRACKING_AUTOMATIC);
        $current = (object)array('completionstate' => COMPLETION_COMPLETE, 'overrideby' => null);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->willReturn($current);
        $c->expects($this->once())
            ->method('internal_get_state')
            ->willReturn(COMPLETION_COMPLETE_PASS);
        $changed = clone($current);
        $changed->timemodified = time();
        $changed->completionstate = COMPLETION_COMPLETE_PASS;
        $c->expects($this->once())
            ->method('internal_set_data');
        $sink = $this->redirectEvents();
        $c->update_state($cm, COMPLETION_COMPLETE_PASS);
        $events = $sink->get_events();
        $this->assertCount(1, $events);

        // Manual tracking, change state by overriding it manually.
        $c = $mockbuilder->getMock();
        $cm = (object)array('id' => 13, 'course' => 42, 'module' => 1,'completion' => COMPLETION_TRACKING_MANUAL);
        $current1 = (object)array('completionstate' => COMPLETION_INCOMPLETE, 'overrideby' => null);
        $current2 = (object)array('completionstate' => COMPLETION_COMPLETE, 'overrideby' => null);
        $c->expects($this->exactly(2))
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);

        $getinvocations = $this->exactly(2);
        $c->expects($getinvocations)
            ->method('get_data')
            ->with($cm, false, 100)
            ->willReturnCallback(function () use ($getinvocations, $current1, $current2) {
                switch ($getinvocations->numberOfInvocations()) {
                    case 1: return $current1;
                    case 2: return $current2;
                    default: $this->fail('Unexpected invocation count');
                }
            });
        $changed1 = clone($current1);
        $changed1->timemodified = time();
        $changed1->completionstate = COMPLETION_COMPLETE;
        $changed1->overrideby = 314159;

        $changed2 = clone($current2);
        $changed2->timemodified = time();
        $changed2->overrideby = null;
        $changed2->completionstate = COMPLETION_INCOMPLETE;

        $setinvocations = $this->exactly(2);
        $c->expects($setinvocations)
            ->method('internal_set_data')
            ->willReturnCallback(function ($comparecm, $comparewith) use (
                $setinvocations,
                $cm
            ): void {
                switch ($setinvocations->numberOfInvocations()) {
                    case 1:
                    case 2:
                        return;
                    default:
                        $this->fail('Unexpected invocation count');
                }
            });
        $c->update_state($cm, COMPLETION_COMPLETE, 100);
        // And confirm that the status can be changed back to incomplete without an override.
        $c->update_state($cm, COMPLETION_INCOMPLETE, 100);


        // Auto, change state via override, incomplete to complete.
        $c = $mockbuilder->getMock();
        $cm = (object)array('id' => 13, 'course' => 42, 'module' => 1, 'completion' => COMPLETION_TRACKING_AUTOMATIC);
        $current = (object)array('completionstate' => COMPLETION_INCOMPLETE, 'overrideby' => null);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->with($cm, false, 100)
            ->willReturn($current);
        $changed = clone($current);
        $changed->timemodified = time();
        $changed->completionstate = COMPLETION_COMPLETE;
        $changed->overrideby = 314159;
        $c->expects($this->once())
            ->method('internal_set_data');

        $c->expects($this->once())
            ->method('internal_get_state')
            ->willReturn(COMPLETION_COMPLETE);
        $c->update_state($cm, COMPLETION_COMPLETE, 100);

        // Now confirm the status can be changed back from complete to incomplete using an override.
        $c = $mockbuilder->getMock();
        $cm = (object)array('id' => 13, 'course' => 42, 'module' => 1, 'completion' => COMPLETION_TRACKING_AUTOMATIC);
        $current = (object)array('completionstate' => COMPLETION_COMPLETE, 'overrideby' => 2);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->with($cm, false, 100)
            ->willReturn($current);
        $changed = clone($current);
        $changed->timemodified = time();
        $changed->completionstate = COMPLETION_INCOMPLETE;
        $changed->overrideby = 314159;
        $c->expects($this->once())
            ->method('internal_set_data');
        $c->update_state($cm, COMPLETION_INCOMPLETE, 100);
    }

    public function test_update_progress() {
        global $DB;
        $this->setup_data();

        self::setUser($this->user);
        $completion = ['completion' => COMPLETION_TRACKING_MANUAL];
        $forum = self::getDataGenerator()->create_module('forum', ['course' => $this->course->id], $completion);
        $cm = get_coursemodule_from_instance('forum', $forum->id);
        $c = new completion_info($this->course);

        // Current user gets their progress set.
        $c->update_progress($cm, 55);
        $progress = $DB->get_field('course_modules_completion', 'progress',
                                   ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(55, $progress);

        // Progress doesn't go backwards for the current user.
        $c->update_progress($cm, 45);
        $progress = $DB->get_field('course_modules_completion', 'progress',
                                   ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(55, $progress);

        // Progress updated to the higher value.
        $c->update_progress($cm, 65);
        $progress = $DB->get_field('course_modules_completion', 'progress',
                                   ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(65, $progress);

        // Check incorrect $progress values.
        try {
            $c->update_progress($cm, 101);
            $this->fail('Expected exception was not thrown');
        } catch (moodle_exception $e) {
            self::assertStringContainsString('Progress value must be between 0 and 100', $e->getMessage());
        }
        $progress = $DB->get_field('course_modules_completion', 'progress',
                                   ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(65, $progress); // Value didn't change.

        try {
            $c->update_progress($cm, -5);
            $this->fail('Expected exception was not thrown');
        } catch (moodle_exception $e) {
            self::assertStringContainsString('Progress value must be between 0 and 100', $e->getMessage());
        }
        $progress = $DB->get_field('course_modules_completion', 'progress',
                                   ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(65, $progress); // Value didn't change.

        // Progress doesn't get updated for another user.
        $newuser = self::getDataGenerator()->create_user();

        $data = new stdClass();
        $data->id = 0;
        $data->userid = $newuser->id;
        $data->coursemoduleid = $cm->id;
        $data->completionstate = COMPLETION_INCOMPLETE;
        $data->progress = 33;
        $data->timemodified = time();
        $data->viewed = COMPLETION_NOT_VIEWED;
        $data->timecompleted = null;
        $data->reaggregate = 0;
        $anotherid = $DB->insert_record('course_modules_completion', $data);

        $c->update_progress($cm, 75);

        $progress1 = $DB->get_field('course_modules_completion', 'progress',
                                  ['coursemoduleid' => $cm->id, 'userid' => $this->user->id]);
        self::local_assert_equals(75, $progress1);

        $progress2 = $DB->get_field('course_modules_completion', 'progress', ['id' => $anotherid]);
        self::local_assert_equals(33, $progress2);

        // Progress doesn't get updated for another activity.
        $forum2 = self::getDataGenerator()->create_module('forum', ['course' => $this->course->id], $completion);
        $cm2 = get_coursemodule_from_instance('forum', $forum2->id);

        $c->update_progress($cm2, 80, $newuser->id);
        $progress1 = $DB->get_field('course_modules_completion', 'progress',
                                  ['coursemoduleid' => $cm2->id, 'userid' => $newuser->id]);
        self::local_assert_equals(80, $progress1);

        $progress2 = $DB->get_field('course_modules_completion', 'progress', ['id' => $anotherid]);
        self::local_assert_equals(33, $progress2);
    }

    /**
     * Data provider for test_internal_get_state().
     *
     * @return array[]
     */
    public static function internal_get_state_provider(): array {
        return [
            'View required, but not viewed yet' => [
                COMPLETION_VIEW_REQUIRED, 1, '', COMPLETION_INCOMPLETE
            ],
            'View not required and not viewed yet' => [
                COMPLETION_VIEW_NOT_REQUIRED, 1, '', COMPLETION_INCOMPLETE
            ],
            'View not required, grade required but no grade yet, $cm->modname not set' => [
                COMPLETION_VIEW_NOT_REQUIRED, 1, 'modname', COMPLETION_INCOMPLETE
            ],
            'View not required, grade required but no grade yet' => [
                COMPLETION_VIEW_NOT_REQUIRED, 1, '', COMPLETION_INCOMPLETE
            ],
            'View not required, grade not required' => [
                COMPLETION_VIEW_NOT_REQUIRED, 0, '', COMPLETION_COMPLETE
            ],
        ];
    }

    /**
     * Test for completion_info::get_state().
     *
     * @dataProvider internal_get_state_provider
     * @param int $completionview
     * @param int $completionusegrade
     * @param string $unsetfield
     * @param int $expectedstate
     */
    public function test_internal_get_state(int $completionview, int $completionusegrade, string $unsetfield, int $expectedstate): void {
        $this->setup_data();

        $assigngenerator = $this->getDataGenerator()->get_plugin_generator('mod_assign');
        $assign = $assigngenerator->create_instance([
            'course' => $this->course->id,
            'completion' => COMPLETION_ENABLED,
            'completionview' => $completionview,
            'completionusegrade' => $completionusegrade,
        ]);

        $userid = $this->user->id;
        $this->setUser($userid);

        $cm = get_coursemodule_from_instance('assign', $assign->id);
        if ($unsetfield) {
            unset($cm->$unsetfield);
        }
        // If view is required, but they haven't viewed it yet.
        $current = (object)['viewed' => COMPLETION_NOT_VIEWED];

        $this->redirectEvents();

        $completioninfo = new completion_info($this->course);
        $this->assertCompletionEquals($expectedstate, $completioninfo->internal_get_state($cm, $userid, $current));
    }

    /**
     * @covers ::set_module_viewed
     */
    public function test_set_module_viewed(): void {
        $this->mock_setup();

        $mockbuilder = $this->getMockBuilder('completion_info');
        $mockbuilder->onlyMethods(array('is_enabled', 'get_data', 'internal_set_data', 'update_state'));
        $mockbuilder->setConstructorArgs(array((object)array('id' => 42)));
        $cm = (object)array('id' => 13, 'course' => 42);

        // Not tracking completion, should do nothing.
        $c = $mockbuilder->getMock();
        $cm->completionview = COMPLETION_VIEW_NOT_REQUIRED;
        $c->set_module_viewed($cm);

        // Tracking completion but completion is disabled, should do nothing.
        $c = $mockbuilder->getMock();
        $cm->completionview = COMPLETION_VIEW_REQUIRED;
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(false);
        $c->set_module_viewed($cm);

        // Now it's enabled, we expect it to get data. If data already has
        // viewed, still do nothing.
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->with($cm, 0)
            ->willReturn((object)array('viewed' => COMPLETION_VIEWED));
        $c->set_module_viewed($cm);

        // OK finally one that hasn't been viewed, now it should set it viewed
        // and update state.
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('get_data')
            ->with($cm, false, 1337)
            ->willReturn((object)array('viewed' => COMPLETION_NOT_VIEWED));
        $c->expects($this->once())
            ->method('internal_set_data')
            ->with($cm, (object)array('viewed' => COMPLETION_VIEWED));
        $c->expects($this->once())
            ->method('update_state')
            ->with($cm, COMPLETION_COMPLETE, 1337);
        $c->set_module_viewed($cm, 1337);
    }

    /**
     * @covers ::count_user_data
     */
    public function test_count_user_data(): void {
        /** @var moodle_database&MockObject $DB */
        global $DB;
        $this->mock_setup();

        $course = (object)array('id' => 13);
        $cm = (object)array('id' => 42);

        $DB->expects($this->once())
            ->method('get_field_sql')
            ->willReturn(666);

        $c = new completion_info($course);
        $this->assertCompletionEquals(666, $c->count_user_data($cm));
    }

    /**
     * @covers ::delete_all_state
     */
    public function test_delete_all_state(): void {
        /** @var moodle_database&MockObject $DB */
        global $DB, $USER;

        // Create a course with auto completion data.
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $page = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionauto);
        $page2 = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionauto);

        $user1 = $this->getDataGenerator()->create_user();
        $this->getDataGenerator()->enrol_user($user1->id, $course->id);

        $c = new completion_info($course);
        $cm = get_coursemodule_from_id('page', $page2->cmid);
        $c->update_state($cm, COMPLETION_COMPLETE, $user1->id);
        $cm = get_coursemodule_from_id('page', $page->cmid);
        $c->update_state($cm, COMPLETION_COMPLETE, $user1->id);
        $USER = $user1;

        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page->cmid));
        $this->assertCount(1, $data);
        $c->delete_all_state($cm);
        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page->cmid));
        $this->assertCount(0, $data);

        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page2->cmid));
        $this->assertCount(1, $data);
    }

    /**
     * @covers ::reset_all_state
     */
    public function test_reset_all_state(): void {
        global $DB, $USER;

        // Create a course with auto completion data.
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $completionauto = array('completion' => COMPLETION_TRACKING_MANUAL);
        $page = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionauto);
        $page2 = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionauto);

        $user1 = $this->getDataGenerator()->create_user();
        $this->getDataGenerator()->enrol_user($user1->id, $course->id);

        $c = new completion_info($course);
        $cm = get_coursemodule_from_id('page', $page2->cmid);
        $c->update_state($cm, COMPLETION_COMPLETE, $user1->id);
        $cm = get_coursemodule_from_id('page', $page->cmid);
        $c->update_state($cm, COMPLETION_COMPLETE, $user1->id);
        $USER = $user1;

        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page->cmid));
        $this->assertCount(1, $data);
        $c->reset_all_state($cm);
        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page->cmid));
        // It's manual tracking so the completion record has been removed
        $this->assertCount(0, $data);

        $data = $DB->get_records('course_modules_completion', array('coursemoduleid' => $page2->cmid));
        $this->assertCount(1, $data);
    }

    /**
     * Data provider for test_get_data().
     *
     * @return array[]
     */
    public static function get_data_provider(): array {
        return [
            'No completion record' => [
                false, true, false, COMPLETION_INCOMPLETE
            ],
            'Not completed' => [
                false, true, true, COMPLETION_INCOMPLETE
            ],
            'Completed' => [
                false, true, true, COMPLETION_COMPLETE
            ],
            'Whole course, complete' => [
                true, true, true, COMPLETION_COMPLETE
            ],
            'Get data for another user, result should be not cached' => [
                false, false, true,  COMPLETION_INCOMPLETE
            ],
            'Get data for another user, including whole course, result should be not cached' => [
                true, false, true,  COMPLETION_INCOMPLETE
            ],
        ];
    }

    /**
     * Tests for completion_info::get_data().
     *
     * @dataProvider get_data_provider
     * @param bool $wholecourse Whole course parameter for get_data().
     * @param bool $sameuser Whether the user calling get_data() is the user itself.
     * @param bool $hasrecord Whether to create a course_modules_completion record.
     * @param int $completion The completion state expected.
     * @covers ::get_data
     */
    public function test_get_data(bool $wholecourse, bool $sameuser, bool $hasrecord, int $completion): void {
        global $DB;

        $this->setup_data();
        $user = $this->user;

        $choicegenerator = $this->getDataGenerator()->get_plugin_generator('mod_choice');
        $choice = $choicegenerator->create_instance([
            'course' => $this->course->id,
            'completion' => COMPLETION_TRACKING_AUTOMATIC,
            'completionview' => true,
            'completionsubmit' => true,
        ]);

        $cm = get_coursemodule_from_instance('choice', $choice->id);

        // Let's manually create a course completion record instead of going through the hoops to complete an activity.
        if ($hasrecord) {
            $cmcompletionrecord = (object)[
                'coursemoduleid' => $cm->id,
                'userid' => $user->id,
                'completionstate' => $completion,
                'overrideby' => null,
                'timemodified' => 0,
            ];
            $DB->insert_record('course_modules_completion', $cmcompletionrecord);
        }

        if (!$sameuser) {
            $this->setAdminUser();
        } else {
            $this->setUser($user);
        }

        // Mock other completion data.
        $completioninfo = new completion_info($this->course);

        $result = $completioninfo->get_data($cm, $wholecourse, $user->id);

        // Course module ID of the returned completion data must match this activity's course module ID.
        $this->assertCompletionEquals($cm->id, $result->coursemoduleid);
        // User ID of the returned completion data must match the user's ID.
        $this->assertCompletionEquals($user->id, $result->userid);
        // The completion state of the returned completion data must match the expected completion state.
        $this->assertCompletionEquals($completion, $result->completionstate);

        // If the user has no completion record, then the default record should be returned.
        if (!$hasrecord) {
            $this->assertCompletionEquals(0, $result->id);
        }

        // Check that we are including relevant completion data for the module.
        if (!$wholecourse) {
            $this->assertTrue(property_exists($result, 'viewed'));
        }
    }

    public function test_internal_set_data() {
        global $DB;
        $this->setup_data();

        $this->setUser($this->user);
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $forum = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionauto);
        $cm = get_coursemodule_from_instance('forum', $forum->id);
        $c = new completion_info($this->course);
        $cache = $this->get_completion_cache();

        // 1) Test with new data.
        $data = new stdClass();
        $data->id = 0;
        $data->userid = $this->user->id;
        $data->coursemoduleid = $cm->id;
        $data->completionstate = COMPLETION_COMPLETE;
        $data->progress = null;
        $data->timemodified = time();
        $data->viewed = COMPLETION_NOT_VIEWED;
        $data->timecompleted = null;
        $data->reaggregate = 0;

        $c->internal_set_data($cm, $data);
        $d1 = $DB->get_field('course_modules_completion', 'id', array('coursemoduleid' => $cm->id));
        $this->local_assert_equals($d1, $data->id);
        $expectedData = [
            'cacherev' => $this->course->cacherev,
            $cm->id => $DB->get_record('course_modules_completion', ['id' => $d1])
        ];
        $this->local_assert_equals($expectedData, $cache->get($this->user->id . '_' . $this->course->id));

        // 2) Test with existing data and for different user (not cached).
        $forum2 = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionauto);
        $cm2 = get_coursemodule_from_instance('forum', $forum2->id);
        $newuser = $this->getDataGenerator()->create_user();
        $data->userid = $newuser->id;
        $d2id = $DB->insert_record('course_modules_completion', $data);

        $d2 = new stdClass();
        $d2->id = $d2id;
        $d2->userid = $newuser->id;
        $d2->coursemoduleid = $cm2->id;
        $d2->completionstate = COMPLETION_COMPLETE;
        $d2->progress = null;
        $d2->timemodified = time();
        $d2->viewed = COMPLETION_NOT_VIEWED;
        $d2->timecompleted = null;
        $d2->reaggregate = 0;
        $c->internal_set_data($cm2, $d2);
        $this->assertFalse($cache->get($newuser->id . '_' . $this->course->id));
    }

    /**
     * Tests internal_set_data in a situation where:
     *
     *   1. The data is not is the caches.
     *   2. The data is in the database.
     *   3. It is not the current user.
     */
    public function test_internal_set_data_doesnt_populate_cache_for_other_user() {
        global $DB;
        $this->setup_data();
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $forum3 = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionauto);
        $cm3 = get_coursemodule_from_instance('forum', $forum3->id);
        $c = new completion_info($this->course);
        $cache = $this->get_completion_cache();
        $newuser2 = $this->getDataGenerator()->create_user();
        $d3 = new stdClass();
        $d3->userid = $newuser2->id;
        $d3->coursemoduleid = $cm3->id;
        $d3->completionstate = COMPLETION_COMPLETE;
        $d3->progress = null;
        $d3->timemodified = time();
        $d3->viewed = COMPLETION_NOT_VIEWED;
        $d3->timecompleted = null;
        $d3->reaggregate = 0;
        $d3->id = $DB->insert_record('course_modules_completion', $d3);
        $c->internal_set_data($cm3, $d3);
        $this->assertFalse($cache->get($newuser2->id . '_' . $this->course->id));
        $data = $c->get_data($cm3, false, $newuser2->id);
        $expectedData = $DB->get_record('course_modules_completion', ['id' => $d3->id]);
        $this->local_assert_equals($expectedData, $data);
        // It should still not be in the cache.
        $this->assertFalse($cache->get($newuser2->id . '_' . $this->course->id));
    }

    /**
     * Tests internal_set_data in a situation where:
     *
     *   1. The data is not is the caches.
     *   2. The data is in the database.
     *   3. It is the current user.
     */
    public function test_internal_set_data_populates_cache_for_current_user() {
        global $DB;
        $this->setup_data();
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $forum3 = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionauto);
        $cm3 = get_coursemodule_from_instance('forum', $forum3->id);
        $c = new completion_info($this->course);
        $cache = $this->get_completion_cache();
        $newuser2 = $this->getDataGenerator()->create_user();
        $this->setUser($newuser2);

        $this->assertFalse($cache->get($newuser2->id . '_' . $this->course->id));

        $d3 = new stdClass();
        $d3->userid = $newuser2->id;
        $d3->coursemoduleid = $cm3->id;
        $d3->completionstate = COMPLETION_COMPLETE;
        $d3->progress = null;
        $d3->timemodified = time();
        $d3->viewed = COMPLETION_NOT_VIEWED;
        $d3->timecompleted = null;
        $d3->reaggregate = 0;
        $d3->id = $DB->insert_record('course_modules_completion', $d3);

        $c->internal_set_data($cm3, $d3);
        $expectedData = [
            'cacherev' => $this->course->cacherev,
            $cm3->id => $DB->get_record('course_modules_completion', ['id' => $d3->id])
        ];
        $this->local_assert_equals($expectedData, $cache->get($newuser2->id . '_' . $this->course->id));

        $data = $c->get_data($cm3, false, $newuser2->id);
        $this->local_assert_equals($expectedData[$cm3->id], $data);

        // It should still be in the cache, and should still match.
        $this->local_assert_equals($expectedData, $cache->get($newuser2->id . '_' . $this->course->id));
    }

    /**
     * @covers ::get_progress_all
     */
    public function test_get_progress_all_few(): void {
        /** @var moodle_database&MockObject $DB */
        global $DB;
        $this->mock_setup();

        $mockbuilder = $this->getMockBuilder('completion_info');
        $mockbuilder->onlyMethods(array('get_tracked_users'));
        $mockbuilder->setConstructorArgs(array((object)array('id' => 42)));
        $c = $mockbuilder->getMock();

        // With few results.
        $c->expects($this->once())
            ->method('get_tracked_users')
            ->with(false,  array(),  0,  '',  '',  '',  null)
            ->willReturn(array(
                (object)array('id' => 100, 'firstname' => 'Woot', 'lastname' => 'Plugh'),
                (object)array('id' => 201, 'firstname' => 'Vroom', 'lastname' => 'Xyzzy')));
        $DB->expects($this->once())
            ->method('get_in_or_equal')
            ->with(array(100, 201))
            ->willReturn(array(' IN (100, 201)', array()));
        $progress1 = (object)array('userid' => 100, 'coursemoduleid' => 13);
        $progress2 = (object)array('userid' => 201, 'coursemoduleid' => 14);
        $DB->expects($this->once())
            ->method('get_recordset_sql')
            ->willReturn(new core_completionlib_fake_recordset(array($progress1, $progress2)));

        $this->assertCompletionEquals(array(
            100 => (object)array('id' => 100, 'firstname' => 'Woot', 'lastname' => 'Plugh',
                'progress' => array(13 => $progress1)),
            201 => (object)array('id' => 201, 'firstname' => 'Vroom', 'lastname' => 'Xyzzy',
                'progress' => array(14 => $progress2)),
        ), $c->get_progress_all(false));
    }

    /**
     * @covers ::get_progress_all
     */
    public function test_get_progress_all_lots(): void {
        /** @var moodle_database&MockObject $DB */
        global $DB;
        $this->mock_setup();

        $mockbuilder = $this->getMockBuilder('completion_info');
        $mockbuilder->onlyMethods(array('get_tracked_users'));
        $mockbuilder->setConstructorArgs(array((object)array('id' => 42)));
        $c = $mockbuilder->getMock();

        $tracked = array();
        $ids = array();
        $progress = array();
        // With more than 1000 results.
        for ($i = 100; $i < 2000; $i++) {
            $tracked[] = (object)array('id' => $i, 'firstname' => 'frog', 'lastname' => $i);
            $ids[] = $i;
            $progress[] = (object)array('userid' => $i, 'coursemoduleid' => 13);
            $progress[] = (object)array('userid' => $i, 'coursemoduleid' => 14);
        }
        $c->expects($this->once())
            ->method('get_tracked_users')
            ->with(true,  3,  0,  '',  '',  '',  null)
            ->willReturn($tracked);
        $inorequalsinvocations = $this->exactly(2);
        $DB->expects($inorequalsinvocations)
            ->method('get_in_or_equal')
            ->willReturnCallback(function ($paramids) use ($inorequalsinvocations, $ids) {
                switch ($inorequalsinvocations->numberOfInvocations()) {
                    case 1:
                        $this->assertCompletionEquals(array_slice($ids, 0, 1000), $paramids);
                        return [' IN whatever', []];
                    case 2:
                        $this->assertCompletionEquals(array_slice($ids, 1000), $paramids);
                        return [' IN whatever2', []];
                    default:
                        $this->fail('Unexpected invocation count');
                }
            });
        $getinvocations = $this->exactly(2);
        $DB->expects($getinvocations)
            ->method('get_recordset_sql')
            ->willReturnCallback(function () use ($getinvocations, $progress) {
                switch ($getinvocations->numberOfInvocations()) {
                    case 1: return new core_completionlib_fake_recordset(array_slice($progress, 0, 1000));
                    case 2: return new core_completionlib_fake_recordset(array_slice($progress, 1000));
                    default: $this->fail('Unexpected invocation count');
                }
            });

        $result = $c->get_progress_all(true, 3);
        $resultok = $ids == array_keys($result);

        foreach ($result as $userid => $data) {
            $resultok = $resultok && $data->firstname == 'frog';
            $resultok = $resultok && $data->lastname == $userid;
            $resultok = $resultok && $data->id == $userid;
            $cms = $data->progress;
            $resultok = $resultok && (array(13, 14) == array_keys($cms));
            $resultok = $resultok && ((object)array('userid' => $userid, 'coursemoduleid' => 13) == $cms[13]);
            $resultok = $resultok && ((object)array('userid' => $userid, 'coursemoduleid' => 14) == $cms[14]);
        }
        $this->assertTrue($resultok);
        $this->assertCount(count($tracked), $result);
    }

    /**
     * @covers ::inform_grade_changed
     */
    public function test_inform_grade_changed(): void {
        $this->mock_setup();

        $mockbuilder = $this->getMockBuilder('completion_info');
        $mockbuilder->onlyMethods(array('is_enabled', 'update_state'));
        $mockbuilder->setConstructorArgs(array((object)array('id' => 42)));

        $cm = (object)array('course' => 42, 'id' => 13, 'completion' => 0, 'completiongradeitemnumber' => null);
        $item = (object)array('itemnumber' => 3,  'gradepass' => 1,  'hidden' => 0);
        $grade = (object)array('userid' => 31337,  'finalgrade' => 0,  'rawgrade' => 0);

        // Not enabled (should do nothing).
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(false);
        $c->inform_grade_changed($cm, $item, $grade, false);

        // Enabled but still no grade completion required,  should still do nothing.
        $c = $mockbuilder->getMock();
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->inform_grade_changed($cm, $item, $grade, false);

        // Enabled and completion required but item number is wrong,  does nothing.
        $c = $mockbuilder->getMock();
        $cm = (object)array('course' => 42, 'id' => 13, 'completion' => 0, 'completiongradeitemnumber' => 7);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->inform_grade_changed($cm, $item, $grade, false);

        // Enabled and completion required and item number right. It is supposed
        // to call update_state with the new potential state being obtained from
        // internal_get_grade_state.
        $c = $mockbuilder->getMock();
        $cm = (object)array('course' => 42, 'id' => 13, 'completion' => 0, 'completiongradeitemnumber' => 3);
        $grade = (object)array('userid' => 31337,  'finalgrade' => 1,  'rawgrade' => 0);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('update_state')
            ->with($cm, COMPLETION_COMPLETE_PASS, 31337)
            ->willReturn(true);
        $c->inform_grade_changed($cm, $item, $grade, false);

        // Same as above but marked deleted. It is supposed to call update_state
        // with new potential state being COMPLETION_INCOMPLETE.
        $c = $mockbuilder->getMock();
        $cm = (object)array('course' => 42, 'id' => 13, 'completion' => 0, 'completiongradeitemnumber' => 3);
        $grade = (object)array('userid' => 31337,  'finalgrade' => 1,  'rawgrade' => 0);
        $c->expects($this->once())
            ->method('is_enabled')
            ->with($cm)
            ->willReturn(true);
        $c->expects($this->once())
            ->method('update_state')
            ->with($cm, COMPLETION_INCOMPLETE, 31337)
            ->willReturn(true);
        $c->inform_grade_changed($cm, $item, $grade, true);
    }

    /**
     * @covers ::internal_get_grade_state
     */
    public function test_internal_get_grade_state(): void {
        $this->mock_setup();

        $item = new stdClass;
        $grade = new stdClass;

        $item->gradepass = 4;
        $item->hidden = 0;
        $grade->rawgrade = 4.0;
        $grade->finalgrade = null;

        // Grade has pass mark and is not hidden,  user passes.
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE_PASS,
            completion_info::internal_get_grade_state($item, $grade));

        // Same but user fails.
        $grade->rawgrade = 3.9;
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE_FAIL,
            completion_info::internal_get_grade_state($item, $grade));

        // User fails on raw grade but passes on final.
        $grade->finalgrade = 4.0;
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE_PASS,
            completion_info::internal_get_grade_state($item, $grade));

        // Item isn't hidden but has no pass mark.
        $item->hidden = 0;
        $item->gradepass = 0;
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE,
            completion_info::internal_get_grade_state($item, $grade));

        // Item is hidden, but returnpassfail is true and the grade is passing.
        $item->hidden = 1;
        $item->gradepass = 4;
        $grade->finalgrade = 5.0;
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE_PASS,
            completion_info::internal_get_grade_state($item, $grade, true));

        // Item is not hidden, but returnpassfail is true and the grade is failing.
        $item->hidden = 0;
        $item->gradepass = 4;
        $grade->finalgrade = 3.0;
        $this->assertCompletionEquals(
            COMPLETION_COMPLETE_FAIL,
            completion_info::internal_get_grade_state($item, $grade, true));
    }

    public function test_get_activities() {
        global $CFG;

        // Enable completion before creating modules, otherwise the completion data is not written in DB.
        $CFG->enablecompletion = true;

        // Create a course with mixed auto completion data.
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $completionmanual = array('completion' => COMPLETION_TRACKING_MANUAL);
        $completionnone = array('completion' => COMPLETION_TRACKING_NONE);
        $forum = $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionauto);
        $page = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionauto);
        $data = $this->getDataGenerator()->create_module('data', array('course' => $course->id), $completionmanual);

        $forum2 = $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionnone);
        $page2 = $this->getDataGenerator()->create_module('page', array('course' => $course->id), $completionnone);
        $data2 = $this->getDataGenerator()->create_module('data', array('course' => $course->id), $completionnone);

        // Create data in another course to make sure it's not considered.
        $course2 = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $this->getDataGenerator()->create_module('forum', array('course' => $course2->id), $completionauto);
        $this->getDataGenerator()->create_module('page', array('course' => $course2->id), $completionmanual);
        $this->getDataGenerator()->create_module('data', array('course' => $course2->id), $completionnone);

        $c = new completion_info($course);
        $activities = $c->get_activities();
        $this->assertCount(3, $activities);
        $this->assertTrue(isset($activities[$forum->cmid]));
        $this->assertSame($forum->name, $activities[$forum->cmid]->name);
        $this->assertTrue(isset($activities[$page->cmid]));
        $this->assertSame($page->name, $activities[$page->cmid]->name);
        $this->assertTrue(isset($activities[$data->cmid]));
        $this->assertSame($data->name, $activities[$data->cmid]->name);

        $this->assertFalse(isset($activities[$forum2->cmid]));
        $this->assertFalse(isset($activities[$page2->cmid]));
        $this->assertFalse(isset($activities[$data2->cmid]));
    }

    public function test_has_activities() {
        global $CFG;

        // Enable completion before creating modules, otherwise the completion data is not written in DB.
        $CFG->enablecompletion = true;

        // Create a course with mixed auto completion data.
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $course2 = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $completionnone = array('completion' => COMPLETION_TRACKING_NONE);
        $this->getDataGenerator()->create_module('forum', array('course' => $course->id), $completionauto);
        $this->getDataGenerator()->create_module('forum', array('course' => $course2->id), $completionnone);

        $c1 = new completion_info($course);
        $c2 = new completion_info($course2);

        $this->assertTrue($c1->has_activities());
        $this->assertFalse($c2->has_activities());
    }

    /**
     * Test course module completion update event.
     */
    public function test_course_module_completion_updated_event() {
        global $USER, $CFG;

        $this->setup_data();

        $this->setAdminUser();

        $completionauto = array('completion' => COMPLETION_TRACKING_AUTOMATIC);
        $forum = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionauto);

        $c = new completion_info($this->course);
        $activities = $c->get_activities();
        $this->local_assert_equals(1, count($activities));
        $this->assertTrue(isset($activities[$forum->cmid]));
        $this->local_assert_equals($activities[$forum->cmid]->name, $forum->name);

        $current = $c->get_data($activities[$forum->cmid], false, $this->user->id);
        $current = (object)$current;
        $current->completionstate = COMPLETION_COMPLETE;
        $current->timemodified = time();
        $current->timecompleted = null;
        $sink = $this->redirectEvents();
        $c->internal_set_data($activities[$forum->cmid], $current);
        $events = $sink->get_events();
        $event = reset($events);
        $this->assertInstanceOf('\core\event\course_module_completion_updated', $event);
        $this->local_assert_equals($forum->cmid, $event->get_record_snapshot('course_modules_completion', $event->objectid)->coursemoduleid);
        $this->local_assert_equals($current, $event->get_record_snapshot('course_modules_completion', $event->objectid));
        $this->local_assert_equals(context_module::instance($forum->cmid), $event->get_context());
        $this->local_assert_equals($USER->id, $event->userid);
        $this->local_assert_equals($this->user->id, $event->relateduserid);
        $this->assertInstanceOf('moodle_url', $event->get_url());
        $this->assertEventLegacyData($current, $event);
    }

    /**
     * Test course completed event.
     */
    public function test_course_completed_event() {
        global $USER;

        $this->setup_data();
        $this->setAdminUser();

        $ccompletion = new completion_completion(array('course' => $this->course->id, 'userid' => $this->user->id));

        // Mark course as complete and get triggered event.
        $sink = $this->redirectEvents();
        $ccompletion->mark_complete();
        $events = $sink->get_events();
        $event = reset($events);

        $this->assertInstanceOf('\core\event\course_completed', $event);
        $this->local_assert_equals($this->course->id, $event->get_record_snapshot('course_completions', $event->objectid)->course);
        $this->local_assert_equals($this->course->id, $event->courseid);
        $this->local_assert_equals($USER->id, $event->userid);
        $this->local_assert_equals($this->user->id, $event->relateduserid);
        $this->local_assert_equals(context_course::instance($this->course->id), $event->get_context());
        $this->assertInstanceOf('moodle_url', $event->get_url());
        $data = $ccompletion->get_record_data();
        $this->assertEventLegacyData($data, $event);
    }

    /**
     * Test course completed event.
     */
    public function test_course_completion_updated_event() {
        $this->setup_data();
        $coursecontext = context_course::instance($this->course->id);
        $coursecompletionevent = \core\event\course_completion_updated::create(
            array(
                'courseid' => $this->course->id,
                'context' => $coursecontext
            )
        );

        // Mark course as complete and get triggered event.
        $sink = $this->redirectEvents();
        $coursecompletionevent->trigger();
        $events = $sink->get_events();
        $event = array_pop($events);
        $sink->close();

        $this->assertInstanceOf('\core\event\course_completion_updated', $event);
        $this->local_assert_equals($this->course->id, $event->courseid);
        $this->local_assert_equals($coursecontext, $event->get_context());
        $this->assertInstanceOf('moodle_url', $event->get_url());
        $expectedlegacylog = array($this->course->id, 'course', 'completion updated', 'completion.php?id='.$this->course->id);
        $this->assertEventLegacyLogData($expectedlegacylog, $event);
    }

    public function test_completion_can_view_data() {
        $this->setup_data();

        $student = $this->getDataGenerator()->create_user();
        $this->getDataGenerator()->enrol_user($student->id, $this->course->id);

        $this->setUser($student);
        $this->assertTrue(completion_can_view_data($student->id, $this->course->id));
        $this->assertFalse(completion_can_view_data($this->user->id, $this->course->id));
    }

    public function test_delete_course_completion_data_including_rpl() {
        global $DB, $USER;

        // Create data, including controls.
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();
        $course1 = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));
        $course2 = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));

        // Course completion.
        $this->getDataGenerator()->enrol_user($user1->id, $course1->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course1->id);
        $this->getDataGenerator()->enrol_user($user1->id, $course2->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course2->id);

        // Criteria completion. Just fake it.
        $sql = "INSERT INTO {course_completion_crit_compl} (userid, course, criteriaid)
                SELECT userid, course, 1
                  FROM {course_completions}";
        $DB->execute($sql);

        // Block stats. Just fake it.
        $DB->delete_records('block_totara_stats');
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_STARTED . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_COMPLETE . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_TIME_SPENT . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);

        // Clear out any logs that might have been created above.
        $DB->delete_records('course_completion_log');

        // Check state of data before running the function.
        $this->local_assert_equals(4, $DB->count_records('course_completions'));
        $this->local_assert_equals(4, $DB->count_records('course_completion_crit_compl'));
        $this->local_assert_equals(12, $DB->count_records('block_totara_stats'));

        // Run the function.
        $completioninfo = new completion_info($course1);
        $completioninfo->delete_course_completion_data_including_rpl();

        // Check that the control data hasn't been affected.
        $this->local_assert_equals(2, $DB->count_records('course_completions'));
        $this->local_assert_equals(2, $DB->count_records('course_completions', array('course' => $course2->id)));

        $this->local_assert_equals(2, $DB->count_records('course_completion_crit_compl'));
        $this->local_assert_equals(2, $DB->count_records('course_completion_crit_compl', array('course' => $course2->id)));

        $this->local_assert_equals(8, $DB->count_records('block_totara_stats'));
        $this->local_assert_equals(6, $DB->count_records('block_totara_stats', array('data2' => $course2->id)));
        $this->local_assert_equals(2, $DB->count_records('block_totara_stats', array('data2' => $course1->id, 'eventtype' => STATS_EVENT_TIME_SPENT)));

        $logs = $DB->get_records('course_completion_log', array(), 'id');
        $this->assertCount(1,$logs);
        $log = reset($logs);

        $this->local_assert_equals(0, $log->userid);
        $this->local_assert_equals($course1->id, $log->courseid);
        $this->local_assert_equals($USER->id, $log->changeuserid);
        $this->assertStringContainsString('Deleted current completion and all crit compl records in delete_course_completion_data_including_rpl', $log->description);
    }

    public function test_delete_course_completion_data() {
        global $DB, $USER;

        // Create data, including controls.
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();
        $course1 = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));
        $course2 = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));

        // Course completion.
        $this->getDataGenerator()->enrol_user($user1->id, $course1->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course1->id);
        $this->getDataGenerator()->enrol_user($user1->id, $course2->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course2->id);

        // Criteria completion. Just fake it.
        $sql = "INSERT INTO {course_completion_crit_compl} (userid, course, criteriaid)
                SELECT userid, course, 1
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "UPDATE {course_completions}
                   SET status = " . COMPLETION_STATUS_COMPLETEVIARPL . "
                 WHERE userid = :userid";
        $DB->execute($sql, array('userid' => $user1->id));

        // Block stats. Just fake it.
        $DB->delete_records('block_totara_stats');
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_STARTED . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_COMPLETE . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_TIME_SPENT . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);

        // Clear out any logs that might have been created above.
        $DB->delete_records('course_completion_log');

        // Check state of data before running the function.
        $this->local_assert_equals(4, $DB->count_records('course_completions'));
        $this->local_assert_equals(4, $DB->count_records('course_completion_crit_compl'));
        $this->local_assert_equals(12, $DB->count_records('block_totara_stats'));

        // Run the function with a userid. This will affect only records for that user, INCLUDING rpl completions.
        $completioninfo = new completion_info($course1);
        $completioninfo->delete_course_completion_data($user1->id);

        // Check that the control data hasn't been affected.
        $this->local_assert_equals(3, $DB->count_records('course_completions'));
        $this->local_assert_equals(2, $DB->count_records('course_completions', array('course' => $course2->id)));
        $this->local_assert_equals(1, $DB->count_records('course_completions', array('course' => $course1->id, 'userid' => $user2->id)));

        $this->local_assert_equals(3, $DB->count_records('course_completion_crit_compl'));
        $this->local_assert_equals(2, $DB->count_records('course_completion_crit_compl', array('course' => $course2->id)));
        $this->local_assert_equals(1, $DB->count_records('course_completion_crit_compl', array('course' => $course1->id, 'userid' => $user2->id)));

        $this->local_assert_equals(10, $DB->count_records('block_totara_stats'));
        $this->local_assert_equals(6, $DB->count_records('block_totara_stats', array('data2' => $course2->id)));
        $this->local_assert_equals(3, $DB->count_records('block_totara_stats', array('data2' => $course1->id, 'userid' => $user2->id)));
        $this->local_assert_equals(1, $DB->count_records('block_totara_stats', array('data2' => $course1->id, 'userid' => $user1->id, 'eventtype' => STATS_EVENT_TIME_SPENT)));

        $logs = $DB->get_records('course_completion_log', array(), 'id');
        $this->assertCount(1,$logs);
        $log = reset($logs);

        $this->local_assert_equals($user1->id, $log->userid);
        $this->local_assert_equals($course1->id, $log->courseid);
        $this->local_assert_equals($USER->id, $log->changeuserid);
        $this->assertStringContainsString('Deleted current completion and all crit compl records in delete_course_completion_data', $log->description);

        // Clear out any logs that might have been created above.
        $DB->delete_records('course_completion_log');

        // Run the function with no userid. This will affect all records for course2, EXCLUDING rpl completions,
        // which means that just user2's course2 records will be affected.
        $completioninfo = new completion_info($course2);
        $completioninfo->delete_course_completion_data();

        // Check that the control data hasn't been affected.
        $this->local_assert_equals(2, $DB->count_records('course_completions'));
        $this->local_assert_equals(1, $DB->count_records('course_completions', array('course' => $course2->id, 'userid' => $user1->id)));
        $this->local_assert_equals(1, $DB->count_records('course_completions', array('course' => $course1->id, 'userid' => $user2->id)));

        $this->local_assert_equals(2, $DB->count_records('course_completion_crit_compl'));
        $this->local_assert_equals(1, $DB->count_records('course_completion_crit_compl', array('course' => $course2->id, 'userid' => $user1->id)));
        $this->local_assert_equals(1, $DB->count_records('course_completion_crit_compl', array('course' => $course1->id, 'userid' => $user2->id)));

        $this->local_assert_equals(8, $DB->count_records('block_totara_stats'));
        $this->local_assert_equals(3, $DB->count_records('block_totara_stats', array('data2' => $course2->id, 'userid' => $user1->id)));
        $this->local_assert_equals(3, $DB->count_records('block_totara_stats', array('data2' => $course1->id, 'userid' => $user2->id)));
        $this->local_assert_equals(1, $DB->count_records('block_totara_stats', array('data2' => $course1->id, 'userid' => $user1->id, 'eventtype' => STATS_EVENT_TIME_SPENT)));
        $this->local_assert_equals(1, $DB->count_records('block_totara_stats', array('data2' => $course2->id, 'userid' => $user2->id, 'eventtype' => STATS_EVENT_TIME_SPENT)));

        $logs = $DB->get_records('course_completion_log', array(), 'id');
        $this->assertCount(1,$logs);
        $log = reset($logs);

        $this->local_assert_equals(0, $log->userid);
        $this->local_assert_equals($course2->id, $log->courseid);
        $this->local_assert_equals($USER->id, $log->changeuserid);
        $this->assertStringContainsString('Deleted current completion and all crit compl records except where the current completion was RPL in delete_course_completion_data', $log->description);
    }

    public function test_delete_course_completion_data_bulk() {
        global $DB;

        // Create data, including controls.
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();
        $course = $this->getDataGenerator()->create_course(array('enablecompletion' => COMPLETION_ENABLED));

        // Course completion.
        $this->getDataGenerator()->enrol_user($user1->id, $course->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course->id);

        // Criteria completion. Just fake it.
        $sql = "INSERT INTO {course_completion_crit_compl} (userid, course, criteriaid)
                SELECT userid, course, 1
                  FROM {course_completions}";
        $DB->execute($sql);

        // Block stats. Just fake it.
        $DB->delete_records('block_totara_stats');
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_STARTED . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_COURSE_COMPLETE . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);
        $sql = "INSERT INTO {block_totara_stats} (userid, timestamp, eventtype, data, data2)
                SELECT userid, 123, " . STATS_EVENT_TIME_SPENT . ", 0, course
                  FROM {course_completions}";
        $DB->execute($sql);

        // Clear out any logs that might have been created above.
        $DB->delete_records('course_completion_log');

        // Check state of data before running the function.
        $this->assertEquals(2, $DB->count_records('course_completions'));
        $this->assertEquals(2, $DB->count_records('course_completion_crit_compl'));
        $this->assertEquals(6, $DB->count_records('block_totara_stats'));

        // Run the function with both userids.
        $userids = [$user1->id, $user2->id];
        $completioninfo = new completion_info($course);
        $completioninfo->delete_course_completion_data_bulk($userids);

        // Check
        $this->assertEquals(0, $DB->count_records('course_completions'));
        $this->assertEquals(0, $DB->count_records('course_completion_crit_compl'));
        $this->assertEquals(2, $DB->count_records('block_totara_stats')); // 2 stats with type STATS_EVENT_TIME_SPENT left

        $logs = $DB->get_records('course_completion_log', array(), 'id');
        $this->assertCount(2, $logs);

        foreach ($logs as $log) {
            $this->assertContains($log->userid, $userids);
            $this->assertEquals($log->courseid, $course->id);
        }
    }

    /**
     * Tests delete_all_completion_data.
     */
    public function test_course_completion_reset() {
        global $DB;

        set_config('enablecompletion', 1);

        $course1 = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $course2 = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();

        $cm1 = $this->getDataGenerator()->create_module('forum', array('course' => $course1->id));
        $cm2 = $this->getDataGenerator()->create_module('forum', array('course' => $course2->id));

        // Fake some module completion records.
        $cmc = new stdClass();
        $cmc->timemodified = 234;

        $cmc->userid = $user1->id;
        $cmc->coursemoduleid = $cm1->id;
        $cmc->completionstate = COMPLETION_STATUS_NOTYETSTARTED;
        $DB->insert_record('course_modules_completion', $cmc);

        $cmc->userid = $user1->id;
        $cmc->coursemoduleid = $cm2->id;
        $cmc->completionstate = COMPLETION_COMPLETE;
        $DB->insert_record('course_modules_completion', $cmc);

        $cmc->userid = $user2->id;
        $cmc->coursemoduleid = $cm1->id;
        $cmc->completionstate = COMPLETION_COMPLETE_PASS;
        $DB->insert_record('course_modules_completion', $cmc);

        $cmc->userid = $user2->id;
        $cmc->coursemoduleid = $cm2->id;
        $cmc->completionstate = COMPLETION_COMPLETE_FAIL;
        $DB->insert_record('course_modules_completion', $cmc);

        // Clear out any existing logs that might have been created.
        $DB->delete_records('course_completion_log');

        // Run the function.
        $completioninfo = new completion_info($course1);
        $completioninfo->delete_all_completion_data();

        // Check that two logs were created, only for the course that was reset.
        $this->local_assert_equals(2, $DB->count_records('course_completion_log'));
        $this->local_assert_equals(2, $DB->count_records('course_completion_log',
            array('courseid' => $course1->id, 'userid' => null)));
    }

    public function test_invalidate_cache_for_single_course(): void {
        // Set up.
        global $CFG;
        $this->setAdminUser();
        // Create 2 courses, each with completion turned on.
        $CFG->enablecompletion = true;
        $CFG->enableavailability = true;
        $generator = $this->getDataGenerator();
        $course1 = $generator->create_course(['enablecompletion' => 1]);
        $course2 = $generator->create_course(['enablecompletion' => 1]);
        // Add an activity to each course.
        $quiz1_cm = $generator->create_module('quiz', ['course' => $course1->id]);
        $activity1 = get_coursemodule_from_id('quiz', $quiz1_cm->cmid);
        $quiz2_cm = $generator->create_module('quiz', ['course' => $course2->id]);
        $activity2 = get_coursemodule_from_id('quiz', $quiz2_cm->cmid);

        // Enrol the test users in different courses.
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();
        $this->getDataGenerator()->enrol_user($user1->id, $course1->id);
        $this->getDataGenerator()->enrol_user($user2->id, $course2->id);

        // For course1 and user1, make sure there are some course completion records *****.
        // Mark activity complete.
        $completion_info1 = new completion_info($course1);
        $completion_info1->update_state($activity1, COMPLETION_COMPLETE, $user1->id);

        // Check that there is a completion cache file existing for course1 & user1.
        $reflection1a = new ReflectionMethod($completion_info1, 'get_progressinfo_cache_key');
        $reflection1a->setAccessible(true);
        $reflection1b = new ReflectionMethod($completion_info1, 'get_progressinfo_cache');
        $reflection1b->setAccessible(true);

        $user1_progressinfo_cache_key = $reflection1a->invoke($completion_info1, $user1->id);
        $this->assertNotEmpty($user1_progressinfo_cache_key);
        $progressinfo_cache1 = $reflection1b->invoke($completion_info1);
        $this->assertNotNull($progressinfo_cache1->phpunitcache[$user1_progressinfo_cache_key]);

        // For course2 and user2, make sure there are some course completion records *****.
        $completion_info1 = new completion_info($course2);
        $completion_info1->update_state($activity2, COMPLETION_COMPLETE, $user2->id);
        // Mark activity complete.
        $completion_info2 = new completion_info($course2);
        $completion_info2->update_state($activity2, COMPLETION_COMPLETE, $user2->id);

        // Check that there is a completion cache file existing for course2 & user2.
        $reflection2a = new ReflectionMethod($completion_info2, 'get_progressinfo_cache_key');
        $reflection2a->setAccessible(true);
        $reflection2b = new ReflectionMethod($completion_info2, 'get_progressinfo_cache');
        $reflection2b->setAccessible(true);

        $user2_progressinfo_cache_key = $reflection2a->invoke($completion_info2, $user2->id);
        $this->assertNotEmpty($user2_progressinfo_cache_key);
        $progressinfo_cache2 = $reflection2b->invoke($completion_info1);
        $this->assertNotNull($progressinfo_cache2->phpunitcache[$user2_progressinfo_cache_key]);

        // Operate.
        // Act as if there was an update, delete or reset course completions operation run for a single course id.
        // Purging the cache files should only happen for course1.
        $completion_info1->invalidatecache($course1->id);

        // This cache key for course1 should have been purged.
        $user1_progressinfo_cache_key = $reflection1a->invoke($completion_info1, $user1->id);
        $progressinfo_cache1 = $reflection1b->invoke($completion_info1); // calling $completion_info1->get_progressinfo_cache()
        $this->assertFalse(array_key_exists($user1_progressinfo_cache_key, $progressinfo_cache1->phpunitcache));

        // This cache key should have been left alone, is for course2.
        $user2_progressinfo_cache_key = $reflection2a->invoke($completion_info2, $user2->id);
        $progressinfo_cache2 = $reflection2b->invoke($completion_info2); // calling $completion_info2->get_progressinfo_cache()
        $this->assertNotNull($progressinfo_cache2->phpunitcache[$user2_progressinfo_cache_key]);
    }

    public static function assertCompletionEquals($expected, $actual, string $message = ''): void {
        // Nasty cheating hack: prevent random failures on timemodified field.
        if (is_array($actual) && (is_object($expected) || is_array($expected))) {
            $actual = (object) $actual;
            $expected = (object) $expected;
        }
        if (is_object($expected) && is_object($actual)) {
            if (property_exists($expected, 'timemodified') && property_exists($actual, 'timemodified')) {
                if ($expected->timemodified + 1 == $actual->timemodified) {
                    $expected = clone($expected);
                    $expected->timemodified = $actual->timemodified;
                }
            }
        }
        parent::assertEquals($expected, $actual, $message);
    }
}

class core_completionlib_fake_recordset implements Iterator {
    protected $closed;
    protected $values, $index;

    public function __construct($values) {
        $this->values = $values;
        $this->index = 0;
    }

    #[\ReturnTypeWillChange]
    public function current() {
        return $this->values[$this->index];
    }

    #[\ReturnTypeWillChange]
    public function key() {
        return $this->values[$this->index];
    }

    #[\ReturnTypeWillChange]
    public function next() {
        $this->index++;
    }

    #[\ReturnTypeWillChange]
    public function rewind() {
        $this->index = 0;
    }

    #[\ReturnTypeWillChange]
    public function valid() {
        return count($this->values) > $this->index;
    }

    public function close() {
        $this->closed = true;
    }

    public function was_closed() {
        return $this->closed;
    }
}
