<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Cody Finegan <cody.finegan@totara.com>
 * @package core_orm
 */

use core\model\encrypted_config;
use core\model\encrypted_config_plugin;

/**
 * Tests covering the config/config_plugin encryption steps.
 *
 * @package core
 * @group orm
 */
class core_orm_encrypted_config_test extends \core_phpunit\testcase {

    /**
     * @return array[]
     */
    public static function model_data_provider(): array {
        return [
            [\core\entity\encrypted_config::class, encrypted_config::class, false],
            [\core\entity\encrypted_config_plugin::class, encrypted_config_plugin::class, true],
        ];
    }

    /**
     * Assert we can store and retrieve a value via the config/config_plugin models.
     *
     * @param string $entity_class
     * @param string $model_class
     * @param bool $plugin
     * @return void
     * @dataProvider model_data_provider
     */
    public function test_encrypted_config(string $entity_class, string $model_class, bool $plugin): void {
        global $DB;

        // Save a value
        $name = 'secret';
        $secret = 'I am a Secret';
        $plugin_name = $plugin ? 'test_plugin' : 'core';

        if ($plugin) {
            encrypted_config_plugin::set_config_value($plugin_name, $name, $secret);
        } else {
            encrypted_config::set_config_value($name, $secret);
        }

        if ($plugin) {
            $raw_value = $DB->get_record('config_plugins', ['plugin' => $plugin_name, 'name' => $name], 'value', MUST_EXIST);
        } else {
            $raw_value = $DB->get_record('config', ['name' => $name], 'value', MUST_EXIST);
        }
        $this->assertNotSame($secret, $raw_value->value);

        // Confirm we can decrypt it
        $value = $plugin ? encrypted_config_plugin::get_config_value($plugin_name, $name) : encrypted_config::get_config_value($name);
        $this->assertEquals($secret, $value);

        $new_secret = 'Updated secret';
        if ($plugin) {
            encrypted_config_plugin::set_config_value($plugin_name, $name, $new_secret);
            $raw_value = $DB->get_record('config_plugins', ['plugin' => $plugin_name, 'name' => $name], 'value', MUST_EXIST);
            $value = encrypted_config_plugin::get_config_value($plugin_name, $name);
        } else {
            encrypted_config::set_config_value($name, $new_secret);
            $raw_value = $DB->get_record('config', ['name' => $name], 'value', MUST_EXIST);
            $value = encrypted_config::get_config_value($name);
        }
        $this->assertNotEquals($new_secret, $raw_value->value);
        $this->assertEquals($new_secret, $value);
    }

}
