<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Produces a sample PDF using lib/pdflib.php
 *
 * @package    core
 * @copyright  2009 David Mudrak <david.mudrak@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require(__DIR__ . '/../../../config.php');
require_once($CFG->libdir . '/pdflib.php');

require_login();
$context = context_system::instance();
require_capability('moodle/site:config', $context);

$getpdf     = optional_param('getpdf', 0, PARAM_INT);
$fontfamily = optional_param('fontfamily', PDF_FONT_NAME_MAIN, PARAM_ALPHA);  // to be configurable

if (!$fontfamily) {
    $fontfamily = PDF_FONT_NAME_MAIN;
}

/**
 * Extend the standard PDF class to get access to some protected values we want to display
 * at the test page.
 *
 * @copyright 2009 David Mudrak <david.mudrak@gmail.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
if ($getpdf) {
    $doc = new pdf();

    $TOTARA = new stdClass();
    require("$CFG->dirroot/version.php");

    $doc->SetTitle('Totara PDF library test');
    $doc->SetAuthor('Totara ' . $TOTARA->release);
    $doc->SetCreator('lib/tests/other/pdflibtestpage.php');
    $doc->SetKeywords('Totara, PDF');
    $doc->SetSubject('This has been generated by Totara as its PDF library test page');
    $doc->SetMargins(15, 30);

    $doc->setPrintHeader(true);
    $doc->setHeaderMargin(10);
    $doc->setHeaderFont(array($fontfamily, 'b', 10));
    $doc->setHeaderData('totara/core/pix/logo.png', 12, $SITE->fullname, $CFG->wwwroot);

    $doc->setPrintFooter(true);
    $doc->setFooterMargin(10);
    $doc->setFooterFont(array($fontfamily, '', 8));

    $doc->AddPage();

    $doc->SetTextColor(255,255,255);
    $doc->SetFillColor(155,176,59);
    $doc->SetFont($fontfamily, 'B', 24);
    $doc->Cell(0, 0, 'Totara PDF library test', 0, 1, 'C', 1);

    $doc->SetFont($fontfamily, '', 12);
    $doc->Ln(6);
    $doc->SetTextColor(0,0,0);

    $c  = '<h3>General information</h3>';
    $c .= 'Totara release: '            . $TOTARA->release . '<br />';
    $c .= 'PDF producer: TCPDF '        . TCPDF_STATIC::getTCPDFVersion() . ' (http://www.tcpdf.org) <br />';
    $c .= 'Font family used: '          . $fontfamily   . '<br />';

    $c .= '<h3>Current settings</h3>';
    $c .= '<table border="1"  cellspacing="0" cellpadding="1">';
    foreach (array('K_PATH_MAIN', 'K_PATH_URL', 'K_PATH_FONTS', 'PDF_FONT_NAME_MAIN', 'K_PATH_CACHE', 'K_PATH_IMAGES', 'K_BLANK_IMAGE',
                        'K_CELL_HEIGHT_RATIO', 'K_SMALL_RATIO', 'PDF_CUSTOM_FONT_PATH', 'PDF_DEFAULT_FONT') as $setting) {
        if (defined($setting)) {
            $c .= '<tr style="font-size: x-small;"><td>' . $setting . '</td><td>' . constant($setting) . '</td></tr>';
        }
    }
    $c .= '</table><br />';

    $c .= '<h3>Available font families</h3>';
    $fontfamilies = $doc->get_font_families();
    $list = array();
    foreach ($fontfamilies as $family => $fonts) {
        $f = $family;
        $spacer = '';
        if ($doc->is_core_font_family($family)) {
            $f .= '<sup>*</sup>';
        } else {
            $spacer = ' ';
        }
        if (count($fonts) > 1) {
            $f .= $spacer . '<i>(' . implode(', ', $fonts) . ')</i>';
        }
        $list[] = $f;
    }
    $c .= implode(', ', $list);
    $c .= '<p><i><small>Note: * Standard core fonts are not embedded in PDF files, PDF viewers are using local fonts.</small></i></p>';

    $c .= '<h3>Installed languages and their alphabets</h3>';
    $languages = array();
    $langdirs = get_list_of_plugins('lang', '', $CFG->dataroot);
    array_unshift($langdirs, 'en');
    foreach ($langdirs as $langdir) {
        $enlangconfig = $CFG->dirroot . '/lang/en/langconfig.php';
        if ('en' == $langdir) {
            $langconfig = $enlangconfig;
        } else {
            $langconfig = $CFG->dataroot . '/lang/' . $langdir . '/langconfig.php';
        }
        // Ignore parents here for now.
        $string = array();
        if (is_readable($langconfig)) {
            include($langconfig);
            if (is_array($string)) {
                $languages[$langdir] = new stdClass();
                $languages[$langdir]->langname = isset($string['thislanguage']) ? $string['thislanguage'] : '(unknown)';
                $languages[$langdir]->alphabet = isset($string['alphabet']) ? '"' . $string['alphabet'] . '"': '(no alphabet defined)';
            }
        }
    }
    $c .= '<dl>';
    foreach ($languages as $langcode => $language) {
        $c .= '<dt>' . $language->langname . ' (' . $langcode . ')</dt>';
        $c .= '<dd>' . $language->alphabet . '</dd>';
    }
    $c .= '</dl>';

    $c .= '<h3>Report builder SVG graph embedding</h3>';
    $svgdata = <<<EOT
<svg width="100%" height="100%" version="1.1" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 800 400" onload="init()" preserveAspectRatio="xMidYMid meet" xmlns="http://www.w3.org/2000/svg"><script type="application/ecmascript">// <![CDATA[
var tooltipOn="", initfns=["ttEvt"], tips={"e2":"3","e3":"3","e4":"3"};
function getE(i){return document.getElementById(i)}
function setattr(i,a,v){i.setAttributeNS(null,a,v);return v}
function newel(e,a){
  var ns='http://www.w3.org/2000/svg', ne=document.createElementNS(ns,e),i;
  for(i in a)
    setattr(ne, i, a[i]);
  return ne;
}
function showhide(e,h){setattr(e,'visibility',h?'visible':'hidden');}
function svgNode(e) {
  var d = e.target.correspondingUseElement || e.target, nn = 'svg';
  while(d.parentNode && d.nodeName != nn)
    d = d.parentNode;
  return d.nodeName == nn ? d : null;
}
function svgCursorCoords(e) {
  var d = svgNode(e), pt;
  if(!d || !d.createSVGPoint || !d.getScreenCTM) {
    return [e.clientX,e.clientY];
  }
  pt = d.createSVGPoint(); pt.x = e.clientX; pt.y = e.clientY;
  pt = pt.matrixTransform(d.getScreenCTM().inverse());
  return [pt.x,pt.y];
}
function tooltip(e,callback,on,param) {
  var tt = getE('tooltip'), rect = getE('ttrect'), shadow = getE('ttshdw'),
    offset = 10, pos = svgCursorCoords(e),
    x = pos[0] + offset, y = pos[1] + offset, inner, brect, bw, bh,
    sw, sh,
    de = svgNode(e);
  if(on && !tt) {
    tt = newel('g',{id:'tooltip',visibility:'visible'});
    rect = newel('rect',{
      stroke: 'black',
      'stroke-width': '1px',
      fill: '#ffffcc',
      width:'10px',height:'10px',
      id: 'ttrect',
      rx:'0px',ry:'0px'
    });
    shadow = newel('rect',{
      fill: '#000',
      opacity: 0.3,
      x:'1.5px',y:'1.5px',
      width:'10px',height:'10px',
      id: 'ttshdw',
      rx:'0px',ry:'0px'
    });
    tt.appendChild(shadow);
    tt.appendChild(rect);
  }
  if(tt) {
    if(on) {
      if(tt.parentNode && tt.parentNode != de)
        tt.parentNode.removeChild(tt);
      de.appendChild(tt);
    }
    showhide(tt,on);
  }
  inner = callback(e,tt,on,param);
  if(inner && on) {
    brect = inner.getBBox();
    bw = Math.ceil(brect.width + 6);
    bh = Math.ceil(brect.height + 6);
    setattr(rect, 'width', bw + 'px');
    setattr(rect, 'height', bh + 'px');
    setattr(inner, 'transform', 'translate(' + (bw / 2) + ',0)');
    if(shadow) {
      setattr(shadow, 'width', (bw + 1) + 'px');
      setattr(shadow, 'height', (bh + 1) + 'px');
    }
    if(bw + x > 797.5) {
      x -= bw + offset * 2;
      x = Math.max(x, 0);
    }
    if(bh + y > 397.5) {
      y -= bh + offset * 2;
      y = Math.max(y, 0);
    }
  }
  on && setattr(tt,'transform','translate('+x+' '+y+')');
  tooltipOn = on ? 1 : 0;
}
function newtext(c){return document.createTextNode(c)}
function texttt(e,tt,on,t){
  var ttt = getE('tooltiptext'), lines, i, ts, xpos;
  if(on) {
    lines = t.split('\\n');
    xpos = '3px';
    if(!ttt) {
      ttt = newel('g', {
        id: 'tooltiptext',
        fill: 'black',
        'font-size': '10px',
        'font-family': 'freeserif',
        'font-weight': 'normal',
        'text-anchor': 'middle'
      });
      tt.appendChild(ttt);
    }
    while(ttt.childNodes.length > 0)
      ttt.removeChild(ttt.childNodes[0]);
    for(i = 0; i < lines.length; ++i) {
      ts = newel('text', { y: (13 * (i + 1)) + 'px' });
      ts.appendChild(newtext(lines[i]));
      ttt.appendChild(ts);
    }
  }
  ttt && showhide(ttt,on);
  return ttt;
}
function finditem(e,list) {
  var l = e.target.correspondingUseElement || e.target, t;
  while(!t && l.parentNode) {
    t = l.id && list[l.id]
    l = l.parentNode;
  }
  return t;
}
function init() {
  if(!document.addEventListener || !initfns)
    return;
  for(var f in initfns)
    eval(initfns[f] + '()');
  initfns = [];
}
function ttEvt() {
  document.addEventListener && document.addEventListener('mousemove',
    function(e) {
      var t = finditem(e,tips);
      if(t || tooltipOn)
        tooltip(e,texttt,t,t);
    },false);
}

setTimeout(function(){init()},20);

// ]]></script><rect width="100%" height="100%" fill="#fff" stroke-width="0px"/><rect width="63.8" x="388.6" y="19" height="340" id="e2" style="fill:#3869B1;"/><rect width="63.8" x="532.4" y="19" height="340" id="e3" style="fill:#3869B1;"/><rect width="63.8" x="676.2" y="19" height="340" id="e4" style="fill:#3869B1;"/><path d="M61 365v-6M204.8 365v-6M348.6 365v-6M492.4 365v-6M636.2 365v-6M780 365v-6M55 359h6M55 342h6M55 325h6M55 308h6M55 291h6M55 274h6M55 257h6M55 240h6M55 223h6M55 206h6M55 189h6M55 172h6M55 155h6M55 138h6M55 121h6M55 104h6M55 87h6M55 70h6M55 53h6M55 36h6M55 19h6" stroke-width="1px" stroke="rgb(0,0,0)"/><g stroke="rgb(0,0,0)" stroke-width="1px"><path d="M56 359h729"/><path d="M61 14v350"/></g><g font-size="12px" font-family="freeserif" fill="rgb(0,0,0)"><g text-anchor="end"><text x="49" y="362.6">0</text><text x="49" y="345.6">0.15</text><text x="49" y="328.6">0.3</text><text x="49" y="311.6">0.45</text><text x="49" y="294.6">0.6</text><text x="49" y="277.6">0.75</text><text x="49" y="260.6">0.9</text><text x="49" y="243.6">1.05</text><text x="49" y="226.6">1.2</text><text x="49" y="209.6">1.35</text><text x="49" y="192.6">1.5</text><text x="49" y="175.6">1.65</text><text x="49" y="158.6">1.8</text><text x="49" y="141.6">1.95</text><text x="49" y="124.6">2.1</text><text x="49" y="107.6">2.25</text><text x="49" y="90.6">2.4</text><text x="49" y="73.6">2.55</text><text x="49" y="56.6">2.7</text><text x="49" y="39.6">2.85</text><text x="49" y="22.6">3</text></g><g text-anchor="middle"><text y="379.4" x="132.9">Guest user  </text><text y="379.4" x="276.7">Admin User</text><text y="379.4" x="420.5">A A</text><text y="379.4" x="564.3">B B</text><text y="379.4" x="708.1">C C</text></g></g></svg>
EOT;
    $c .= '<div><img src="@' . base64_encode($svgdata) . '" width="800" height="400"/></div>';

    // Add in a QR code
    $c .= '<h3>QR Code embedding</h3>';
    $qr = pdf::qr('https://example.com', 10, 10);
    $c .= '<div><img src="data:image/png;base64, '. $qr .'" width="150" height="150" /></div>';

    $doc->writeHTML($c);

    $doc->Output('pdflibtestpage.pdf');
    exit();
}

$PAGE->set_url('/lib/tests/other/pdflibtestpage.php');
$PAGE->set_context($context);
$PAGE->set_title('PDF library test');
$PAGE->set_heading('PDF library test');

echo $OUTPUT->header();
echo $OUTPUT->page_main_heading('Press the button to generate test PDF');
echo $OUTPUT->continue_button(new moodle_url($PAGE->url, array('getpdf' => 1, 'fontfamily' => PDF_FONT_NAME_MAIN)));
echo $OUTPUT->footer();
