<?php

// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Various upgrade/install related functions and classes.
 *
 * @package    core
 * @subpackage upgrade
 * @copyright  1999 onwards Martin Dougiamas (http://dougiamas.com)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

/** UPGRADE_LOG_NORMAL = 0 */
define('UPGRADE_LOG_NORMAL', 0);
/** UPGRADE_LOG_NOTICE = 1 */
define('UPGRADE_LOG_NOTICE', 1);
/** UPGRADE_LOG_ERROR = 2 */
define('UPGRADE_LOG_ERROR',  2);

/**
 * Exception indicating unknown error during upgrade.
 *
 * @package    core
 * @subpackage upgrade
 * @copyright  2009 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class upgrade_exception extends moodle_exception {
    function __construct($plugin, $version, $debuginfo=NULL) {
        global $CFG;
        $a = (object)array('plugin'=>$plugin, 'version'=>$version);
        parent::__construct('upgradeerror', 'admin', "$CFG->wwwroot/$CFG->admin/index.php", $a, $debuginfo);
    }
}

/**
 * Exception indicating invalid call to upgrade_main_savepoint() during upgrade.
 */
class upgrade_main_savepoint_exception extends moodle_exception {
    public function __construct($pluginfile) {
        parent::__construct('upgradeerrormainsavepoint', 'admin', '', $pluginfile, $pluginfile);
    }
}

/**
 * Exception indicating downgrade error during upgrade.
 *
 * @package    core
 * @subpackage upgrade
 * @copyright  2009 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class downgrade_exception extends moodle_exception {
    function __construct($plugin, $oldversion, $newversion) {
        global $CFG;
        $plugin = is_null($plugin) ? 'moodle' : $plugin;
        $a = (object)array('plugin'=>$plugin, 'oldversion'=>$oldversion, 'newversion'=>$newversion);
        parent::__construct('cannotdowngrade', 'debug', "$CFG->wwwroot/$CFG->admin/index.php", $a);
    }
}

/**
 * @package    core
 * @subpackage upgrade
 * @copyright  2009 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class upgrade_requires_exception extends moodle_exception {
    function __construct($plugin, $pluginversion, $currentmoodle, $requiremoodle) {
        global $CFG;
        $a = new stdClass();
        $a->pluginname     = $plugin;
        $a->pluginversion  = $pluginversion;
        $a->currentmoodle  = $currentmoodle;
        $a->requiremoodle  = $requiremoodle;
        parent::__construct('pluginrequirementsnotmet', 'error', "$CFG->wwwroot/$CFG->admin/index.php", $a);
    }
}

/**
 * @package    core
 * @subpackage upgrade
 * @copyright  2009 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class plugin_defective_exception extends moodle_exception {
    function __construct($plugin, $details) {
        global $CFG;
        parent::__construct('detectedbrokenplugin', 'error', "$CFG->wwwroot/$CFG->admin/index.php", $plugin, $details);
    }
}

/**
 * Misplaced plugin exception.
 *
 * Note: this should be used only from the upgrade/admin code.
 *
 * @package    core
 * @subpackage upgrade
 * @copyright  2009 Petr Skoda {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class plugin_misplaced_exception extends moodle_exception {
    /**
     * Constructor.
     * @param string $component the component from version.php
     * @param string $expected expected directory, null means calculate
     * @param string $current plugin directory path
     */
    public function __construct($component, $expected, $current) {
        global $CFG;
        if (empty($expected)) {
            list($type, $plugin) = core_component::normalize_component($component);
            $plugintypes = core_component::get_plugin_types();
            if (isset($plugintypes[$type])) {
                $expected = $plugintypes[$type] . '/' . $plugin;
            }
        }
        if (strpos($expected, '$CFG->dirroot') !== 0) {
            $expected = str_replace($CFG->dirroot, '$CFG->dirroot', $expected);
        }
        if (strpos($current, '$CFG->dirroot') !== 0) {
            $current = str_replace($CFG->dirroot, '$CFG->dirroot', $current);
        }
        $a = new stdClass();
        $a->component = $component;
        $a->expected  = $expected;
        $a->current   = $current;
        parent::__construct('detectedmisplacedplugin', 'core_plugin', "$CFG->wwwroot/$CFG->admin/index.php", $a);
    }
}

/**
 * Static class monitors performance of upgrade steps.
 */
class core_upgrade_time {
    /** @var float Time at start of current upgrade (plugin/system) */
    protected static $before;
    /** @var float Time at end of last savepoint */
    protected static $lastsavepoint;
    /** @var bool Flag to indicate whether we are recording timestamps or not. */
    protected static $isrecording = false;

    /**
     * Records current time at the start of the current upgrade item, e.g. plugin.
     */
    public static function record_start() {
        self::$before = microtime(true);
        self::$lastsavepoint = self::$before;
        self::$isrecording = true;
    }

    /**
     * Records current time at the end of a given numbered step.
     *
     * @param float $version Version number (may have decimals, or not)
     */
    public static function record_savepoint($version) {
        global $CFG, $OUTPUT;

        // In developer debug mode we show a notification after each individual save point.
        if ($CFG->debugdeveloper && self::$isrecording) {
            $time = microtime(true);

            $notification = new \core\output\notification($version . ': ' .
                    get_string('successduration', '', format_float($time - self::$lastsavepoint, 2)),
                    \core\output\notification::NOTIFY_SUCCESS);
            $notification->set_show_closebutton(false);
            echo $OUTPUT->render($notification);
            self::$lastsavepoint = $time;
        }
    }

    /**
     * Gets the time since the record_start function was called, rounded to 2 digits.
     *
     * @return float Elapsed time
     */
    public static function get_elapsed() {
        return microtime(true) - self::$before;
    }
}

/**
 * Sets maximum expected time needed for upgrade task.
 * Please always make sure that upgrade will not run longer!
 *
 * The script may be automatically aborted if upgrade times out.
 *
 * @category upgrade
 * @param int $max_execution_time in seconds (can not be less than 60 s)
 */
function upgrade_set_timeout($max_execution_time=300) {
    global $CFG;

    if (!isset($CFG->upgraderunning) or $CFG->upgraderunning < time()) {
        $upgraderunning = get_config(null, 'upgraderunning');
    } else {
        $upgraderunning = $CFG->upgraderunning;
    }

    if (!$upgraderunning) {
        if (CLI_SCRIPT) {
            // never stop CLI upgrades
            $upgraderunning = 0;
        } else {
            // web upgrade not running or aborted
            print_error('upgradetimedout', 'admin', "$CFG->wwwroot/$CFG->admin/");
        }
    }

    if ($max_execution_time < 60) {
        // protection against 0 here
        $max_execution_time = 60;
    }

    $expected_end = time() + $max_execution_time;

    if ($expected_end < $upgraderunning + 10 and $expected_end > $upgraderunning - 10) {
        // no need to store new end, it is nearly the same ;-)
        return;
    }

    if (CLI_SCRIPT) {
        // there is no point in timing out of CLI scripts, admins can stop them if necessary
        core_php_time_limit::raise();
    } else {
        core_php_time_limit::raise($max_execution_time);
    }
    set_config('upgraderunning', $expected_end); // keep upgrade locked until this time
}

/**
 * Upgrade savepoint, marks end of each upgrade block.
 * It stores new main version, resets upgrade timeout
 * and abort upgrade if user cancels page loading.
 *
 * Please do not make large upgrade blocks with lots of operations,
 * for example when adding tables keep only one table operation per block.
 *
 * @category upgrade
 * @param bool $result false if upgrade step failed, true if completed
 * @param string or float $version main version
 * @param bool $allowabort allow user to abort script execution here
 * @return void
 */
function upgrade_main_savepoint($result, $version, $allowabort=true) {
    global $CFG;

    //sanity check to avoid confusion with upgrade_mod_savepoint usage.
    if (!is_bool($allowabort)) {
        $errormessage = 'Parameter type mismatch. Are you mixing up upgrade_main_savepoint() and upgrade_mod_savepoint()?';
        throw new coding_exception($errormessage);
    }

    if (!$result) {
        throw new upgrade_exception(null, $version);
    }

    // Main savepoint may be called from lib/db/upgrade.php and lib/upgradelib.php only.
    $debuginfo = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 1);
    $callee = str_replace(DIRECTORY_SEPARATOR, '/', $debuginfo[0]['file']);
    $dirroot = str_replace(DIRECTORY_SEPARATOR, '/', $CFG->dirroot);
    if ($callee !== $dirroot . '/lib/db/upgrade.php' && $callee !== $dirroot . '/lib/upgradelib.php') {
        throw new upgrade_main_savepoint_exception($debuginfo[0]['file']);
    }

    if ($CFG->version >= $version) {
        // something really wrong is going on in main upgrade script
        throw new downgrade_exception(null, $CFG->version, $version);
    }

    set_config('version', $version);
    upgrade_log(UPGRADE_LOG_NORMAL, null, 'Upgrade savepoint reached');

    // reset upgrade timeout to default
    upgrade_set_timeout();

    core_upgrade_time::record_savepoint($version);

    // this is a safe place to stop upgrades if user aborts page loading
    if ($allowabort and connection_aborted()) {
        die;
    }
}

/**
 * Module upgrade savepoint, marks end of module upgrade blocks
 * It stores module version, resets upgrade timeout
 * and abort upgrade if user cancels page loading.
 *
 * @category upgrade
 * @param bool $result false if upgrade step failed, true if completed
 * @param string or float $version module version
 * @param string $modname name of module
 * @param bool $allowabort allow user to abort script execution here
 * @return void
 */
function upgrade_mod_savepoint($result, $version, $modname, $allowabort=true) {
    upgrade_plugin_savepoint($result, $version, 'mod', $modname, $allowabort);
}

/**
 * Blocks upgrade savepoint, marks end of blocks upgrade blocks
 * It stores block version, resets upgrade timeout
 * and abort upgrade if user cancels page loading.
 *
 * @category upgrade
 * @param bool $result false if upgrade step failed, true if completed
 * @param string or float $version block version
 * @param string $blockname name of block
 * @param bool $allowabort allow user to abort script execution here
 * @return void
 */
function upgrade_block_savepoint($result, $version, $blockname, $allowabort=true) {
    upgrade_plugin_savepoint($result, $version, 'block', $blockname, $allowabort);
}

/**
 * Plugins upgrade savepoint, marks end of blocks upgrade blocks
 * It stores plugin version, resets upgrade timeout
 * and abort upgrade if user cancels page loading.
 *
 * @category upgrade
 * @param bool $result false if upgrade step failed, true if completed
 * @param string|float $version plugin version
 * @param string $type type of plugin
 * @param string $plugin name of plugin
 * @param bool $allowabort allow user to abort script execution here
 * @return void
 */
function upgrade_plugin_savepoint($result, $version, $type, $plugin, $allowabort=true) {
    global $DB;

    if (!\core_component::is_valid_plugin_name($type, $plugin)) {
        throw new coding_exception('Invalid plugin name format');
    }
    $plugins =\core_component::get_plugin_list($type);
    if (!isset($plugins[$plugin])) {
        throw new coding_exception('Invalid plugin name');
    }

    $plugindir = $plugins[$plugin];
    $component = $type.'_'.$plugin;

    if ($type === 'mod') {
        if (!$DB->record_exists('modules', array('name' => $plugin))) {
            print_error('modulenotexist', 'debug', '', $plugin);
        }
    } else if ($type === 'block') {
        if (!$DB->record_exists('block', array('name' => $plugin))) {
            print_error('blocknotexist', 'debug', '', $plugin);
        }
    }

    if (!PHPUNIT_TEST and debugging('', DEBUG_DEVELOPER)) {
        // Unfortunately developers often copy/paste wrong parameters for this method,
        // so make sure to give them a warning when this function is called from wrong upgrade script.
        $upgradescript = $plugindir . '/db/upgrade.php';
        $bt = debug_backtrace();
        if (isset($bt[1]) and (
                $bt[1]['function'] === 'upgrade_plugins' or
                $bt[1]['function'] === 'upgrade_plugins_modules' or
                $bt[1]['function'] === 'upgrade_plugins_blocks')) {
            // All good, this is a version bump without upgrade stuff.

        } else if (file_exists($upgradescript)) {
            $upgradescript = realpath($upgradescript);
            $found = false;
            foreach ($bt as $i => $trace) {
                if (realpath($trace['file']) === $upgradescript) {
                    $found = true;
                    break;
                }
            }
            if (!$found) {
                debugging('savepoint was not called from appropriate upgrade.php file', DEBUG_DEVELOPER);
            }

        } else {
            debugging("db/upgrade.php script for plugin $component does not exist!");
        }
    }

    if (!$result) {
        throw new upgrade_exception($component, $version);
    }

    $dbversion = $DB->get_field('config_plugins', 'value', array('plugin'=>$component, 'name'=>'version'));

    if ($dbversion >= $version) {
        // Something really wrong is going on in the upgrade script
        throw new downgrade_exception($component, $dbversion, $version);
    }

    // Totara: make sure we do not set future version.
    $getconfigversion = function ($versionpath) {
        $plugin = new stdClass();
        $plugin->version = null;
        $module = $plugin;
        include($versionpath);
        return $plugin->version;
    };
    $configversion = $getconfigversion($plugindir . '/version.php');
    if ($version > $configversion) {
        throw new upgrade_exception($component, $version, 'upgrade tried to set version higher than version.php');
    }

    set_config('version', $version, $component);
    upgrade_log(UPGRADE_LOG_NORMAL, $component, 'Upgrade savepoint reached');

    // Reset upgrade timeout to default
    upgrade_set_timeout();

    core_upgrade_time::record_savepoint($version);

    // This is a safe place to stop upgrades if user aborts page loading
    if ($allowabort and connection_aborted()) {
        die;
    }
}

/**
 * Detect if there are leftovers in PHP source files.
 *
 * During main version upgrades administrators MUST move away
 * old PHP source files and start from scratch (or better
 * use git).
 *
 * @return bool true means borked upgrade, false means previous PHP files were properly removed
 */
function upgrade_stale_php_files_present() {
    global $CFG;

    $someexamplesofremovedfiles = array(
        // Removed in Totara 14.0
        '/lib/db/premigrate.php',
    );

    foreach ($someexamplesofremovedfiles as $file) {
        if (file_exists($CFG->dirroot.$file)) {
            return true;
        }
    }

    return false;
}

/**
 * Upgrade plugins
 * @param string $type The type of plugins that should be updated (e.g. 'enrol', 'qtype')
 * return void
 */
function upgrade_plugins($type, $startcallback, $endcallback, $verbose) {
    global $CFG, $DB;

/// special cases
    if ($type === 'mod') {
        return upgrade_plugins_modules($startcallback, $endcallback, $verbose);
    } else if ($type === 'block') {
        return upgrade_plugins_blocks($startcallback, $endcallback, $verbose);
    }

    if ($type === 'totaracoreonly') {
        $type = 'totara';
        $plugs = array('core' => realpath("$CFG->dirroot/totara/core"));
        // Note we can safely run the totara_core upgrade again later.
    } else {
        $plugs = core_component::get_plugin_list($type);
    }

    foreach ($plugs as $plug=>$fullplug) {
        // Reset time so that it works when installing a large number of plugins
        core_php_time_limit::raise(600);
        $component = clean_param($type.'_'.$plug, PARAM_COMPONENT); // standardised plugin name

        // check plugin dir is valid name
        if (empty($component)) {
            throw new plugin_defective_exception($type.'_'.$plug, 'Invalid plugin directory name.');
        }

        if (!is_readable($fullplug.'/version.php')) {
            continue;
        }

        $plugin = new stdClass();
        $plugin->version = null;
        $module = $plugin; // Prevent some notices when plugin placed in wrong directory.
        require($fullplug.'/version.php');  // defines $plugin with version etc
        unset($module);

        if (empty($plugin->version)) {
            throw new plugin_defective_exception($component, 'Missing $plugin->version number in version.php.');
        }

        if (empty($plugin->component)) {
            throw new plugin_defective_exception($component, 'Missing $plugin->component declaration in version.php.');
        }

        if ($plugin->component !== $component) {
            throw new plugin_misplaced_exception($plugin->component, null, $fullplug);
        }

        $plugin->name     = $plug;
        $plugin->fullname = $component;

        if (!empty($plugin->requires)) {
            if ($plugin->requires > $CFG->version) {
                throw new upgrade_requires_exception($component, $plugin->version, $CFG->version, $plugin->requires);
            } else if ($plugin->requires < 2010000000) {
                throw new plugin_defective_exception($component, 'Plugin is not compatible with Moodle 2.x or later.');
            }
        }

        // try to recover from interrupted install.php if needed
        if (file_exists($fullplug.'/db/install.php')) {
            if (get_config($plugin->fullname, 'installrunning')) {
                require_once($fullplug.'/db/install.php');
                $recover_install_function = 'xmldb_'.$plugin->fullname.'_install_recovery';
                if (function_exists($recover_install_function)) {
                    $startcallback($component, true, $verbose);
                    $recover_install_function();
                    unset_config('installrunning', $plugin->fullname);
                    update_capabilities($component);
                    log_update_descriptions($component);
                    external_update_descriptions($component);
                    events_update_definition($component);
                    \core\task\manager::reset_scheduled_tasks_for_component($component);
                    message_update_providers($component);
                    \core\message\inbound\manager::update_handlers_for_component($component);
                    if ($type === 'message') {
                        message_update_processors($plug);
                    }
                    core_tag_area::reset_definitions_for_component($component);
                    $endcallback($component, true, $verbose);
                }
            }
        }

        $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!
        if (empty($installedversion)) { // new installation
            $startcallback($component, true, $verbose);

        /// Install tables if defined
            if (file_exists($fullplug.'/db/install.xml')) {
                $DB->get_manager()->install_from_xmldb_file($fullplug.'/db/install.xml');
            }

        /// store version
            upgrade_plugin_savepoint(true, $plugin->version, $type, $plug, false);

        /// execute post install file
            if (file_exists($fullplug.'/db/install.php')) {
                require_once($fullplug.'/db/install.php');
                set_config('installrunning', 1, $plugin->fullname);
                $post_install_function = 'xmldb_'.$plugin->fullname.'_install';
                $post_install_function();
                unset_config('installrunning', $plugin->fullname);
            }

        /// Install various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            if ($type === 'message') {
                message_update_processors($plug);
            }
            core_tag_area::reset_definitions_for_component($component);
            $endcallback($component, true, $verbose);

        } else if ($installedversion < $plugin->version) { // upgrade
        /// Run the upgrade function for the plugin.
            $startcallback($component, false, $verbose);

            if (is_readable($fullplug.'/db/upgrade.php')) {
                require_once($fullplug.'/db/upgrade.php');  // defines upgrading function

                $newupgrade_function = 'xmldb_'.$plugin->fullname.'_upgrade';
                $result = $newupgrade_function($installedversion);
            } else {
                $result = true;
            }

            $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!
            if ($installedversion < $plugin->version) {
                // store version if not already there
                upgrade_plugin_savepoint($result, $plugin->version, $type, $plug, false);
            }

        /// Upgrade various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            if ($type === 'message') {
                // Ugly hack!
                message_update_processors($plug);
            }
            core_tag_area::reset_definitions_for_component($component);
            $endcallback($component, false, $verbose);

        } else if ($installedversion > $plugin->version) {
            throw new downgrade_exception($component, $installedversion, $plugin->version);
        }
    }
}

/**
 * Find and check all modules and load them up or upgrade them if necessary
 *
 * @global object
 * @global object
 */
function upgrade_plugins_modules($startcallback, $endcallback, $verbose) {
    global $CFG, $DB;

    $mods = core_component::get_plugin_list('mod');

    foreach ($mods as $mod=>$fullmod) {

        if ($mod === 'NEWMODULE') {   // Someone has unzipped the template, ignore it
            continue;
        }

        $component = clean_param('mod_'.$mod, PARAM_COMPONENT);

        // check module dir is valid name
        if (empty($component)) {
            throw new plugin_defective_exception('mod_'.$mod, 'Invalid plugin directory name.');
        }

        if (!is_readable($fullmod.'/version.php')) {
            throw new plugin_defective_exception($component, 'Missing version.php');
        }

        $module = new stdClass();
        $plugin = new stdClass();
        $plugin->version = null;
        require($fullmod .'/version.php');  // Defines $plugin with version etc.

        // Check if the legacy $module syntax is still used.
        if (!is_object($module) or (count((array)$module) > 0)) {
            throw new plugin_defective_exception($component, 'Unsupported $module syntax detected in version.php');
        }

        // Prepare the record for the {modules} table.
        $module = clone($plugin);
        unset($module->version);
        unset($module->component);
        unset($module->dependencies);
        unset($module->release);

        if (empty($plugin->version)) {
            throw new plugin_defective_exception($component, 'Missing $plugin->version number in version.php.');
        }

        if (empty($plugin->component)) {
            throw new plugin_defective_exception($component, 'Missing $plugin->component declaration in version.php.');
        }

        if ($plugin->component !== $component) {
            throw new plugin_misplaced_exception($plugin->component, null, $fullmod);
        }

        if (!empty($plugin->requires)) {
            if ($plugin->requires > $CFG->version) {
                throw new upgrade_requires_exception($component, $plugin->version, $CFG->version, $plugin->requires);
            } else if ($plugin->requires < 2010000000) {
                throw new plugin_defective_exception($component, 'Plugin is not compatible with Moodle 2.x or later.');
            }
        }

        if (empty($module->cron)) {
            $module->cron = 0;
        }

        // all modules must have en lang pack
        if (!is_readable("$fullmod/lang/en/$mod.php")) {
            throw new plugin_defective_exception($component, 'Missing mandatory en language pack.');
        }

        $module->name = $mod;   // The name MUST match the directory

        $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!

        if (file_exists($fullmod.'/db/install.php')) {
            if (get_config($module->name, 'installrunning')) {
                require_once($fullmod.'/db/install.php');
                $recover_install_function = 'xmldb_'.$module->name.'_install_recovery';
                if (function_exists($recover_install_function)) {
                    $startcallback($component, true, $verbose);
                    $recover_install_function();
                    unset_config('installrunning', $module->name);
                    // Install various components too
                    update_capabilities($component);
                    log_update_descriptions($component);
                    external_update_descriptions($component);
                    events_update_definition($component);
                    \core\task\manager::reset_scheduled_tasks_for_component($component);
                    message_update_providers($component);
                    \core\message\inbound\manager::update_handlers_for_component($component);
                    core_tag_area::reset_definitions_for_component($component);
                    $endcallback($component, true, $verbose);
                }
            }
        }

        if (empty($installedversion)) {
            $startcallback($component, true, $verbose);

        /// Execute install.xml (XMLDB) - must be present in all modules
            $DB->get_manager()->install_from_xmldb_file($fullmod.'/db/install.xml');

        /// Add record into modules table - may be needed in install.php already
            $module->id = $DB->insert_record('modules', $module);
            upgrade_plugin_savepoint(true, $plugin->version, 'mod', $module->name, false);

        /// Post installation hook - optional
            if (file_exists("$fullmod/db/install.php")) {
                require_once("$fullmod/db/install.php");
                // Set installation running flag, we need to recover after exception or error
                set_config('installrunning', 1, $module->name);
                $post_install_function = 'xmldb_'.$module->name.'_install';
                $post_install_function();
                unset_config('installrunning', $module->name);
            }

        /// Install various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            core_tag_area::reset_definitions_for_component($component);

            $endcallback($component, true, $verbose);

        } else if ($installedversion < $plugin->version) {
        /// If versions say that we need to upgrade but no upgrade files are available, notify and continue
            $startcallback($component, false, $verbose);

            if (is_readable($fullmod.'/db/upgrade.php')) {
                require_once($fullmod.'/db/upgrade.php');  // defines new upgrading function
                $newupgrade_function = 'xmldb_'.$module->name.'_upgrade';
                $result = $newupgrade_function($installedversion, $module);
            } else {
                $result = true;
            }

            $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!
            $currmodule = $DB->get_record('modules', array('name'=>$module->name));
            if ($installedversion < $plugin->version) {
                // store version if not already there
                upgrade_plugin_savepoint($result, $plugin->version, 'mod', $mod, false);
            }

            // update cron flag if needed
            if ($currmodule->cron != $module->cron) {
                $DB->set_field('modules', 'cron', $module->cron, array('name' => $module->name));
            }

            // Upgrade various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            core_tag_area::reset_definitions_for_component($component);

            $endcallback($component, false, $verbose);

        } else if ($installedversion > $plugin->version) {
            throw new downgrade_exception($component, $installedversion, $plugin->version);
        }
    }
}


/**
 * This function finds all available blocks and install them
 * into blocks table or do all the upgrade process if newer.
 *
 * @global object
 * @global object
 */
function upgrade_plugins_blocks($startcallback, $endcallback, $verbose) {
    global $CFG, $DB;

    require_once($CFG->dirroot.'/blocks/moodleblock.class.php');

    $blocktitles   = array(); // we do not want duplicate titles

    //Is this a first install
    $first_install = null;

    $blocks = core_component::get_plugin_list('block');

    foreach ($blocks as $blockname=>$fullblock) {

        if (is_null($first_install)) {
            $first_install = ($DB->count_records('block_instances') == 0);
        }

        if ($blockname === 'NEWBLOCK') {   // Someone has unzipped the template, ignore it
            continue;
        }

        $component = clean_param('block_'.$blockname, PARAM_COMPONENT);

        // check block dir is valid name
        if (empty($component)) {
            throw new plugin_defective_exception('block_'.$blockname, 'Invalid plugin directory name.');
        }

        if (!is_readable($fullblock.'/version.php')) {
            throw new plugin_defective_exception('block/'.$blockname, 'Missing version.php file.');
        }
        $plugin = new stdClass();
        $plugin->version = null;
        $plugin->cron    = 0;
        $module = $plugin; // Prevent some notices when module placed in wrong directory.
        include($fullblock.'/version.php');
        unset($module);
        $block = clone($plugin);
        unset($block->version);
        unset($block->component);
        unset($block->dependencies);
        unset($block->release);

        if (empty($plugin->version)) {
            throw new plugin_defective_exception($component, 'Missing block version number in version.php.');
        }

        if (empty($plugin->component)) {
            throw new plugin_defective_exception($component, 'Missing $plugin->component declaration in version.php.');
        }

        if ($plugin->component !== $component) {
            throw new plugin_misplaced_exception($plugin->component, null, $fullblock);
        }

        if (!empty($plugin->requires)) {
            if ($plugin->requires > $CFG->version) {
                throw new upgrade_requires_exception($component, $plugin->version, $CFG->version, $plugin->requires);
            } else if ($plugin->requires < 2010000000) {
                throw new plugin_defective_exception($component, 'Plugin is not compatible with Moodle 2.x or later.');
            }
        }

        if (!is_readable($fullblock.'/block_'.$blockname.'.php')) {
            throw new plugin_defective_exception('block/'.$blockname, 'Missing main block class file.');
        }
        include_once($fullblock.'/block_'.$blockname.'.php');

        $classname = 'block_'.$blockname;

        if (!class_exists($classname)) {
            throw new plugin_defective_exception($component, 'Can not load main class.');
        }

        $blockobj    = new $classname;   // This is what we'll be testing
        $blocktitle  = $blockobj->get_title();

        // OK, it's as we all hoped. For further tests, the object will do them itself.
        if (!$blockobj->_self_test()) {
            throw new plugin_defective_exception($component, 'Self test failed.');
        }

        $block->name     = $blockname;   // The name MUST match the directory

        $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!

        if (file_exists($fullblock.'/db/install.php')) {
            if (get_config('block_'.$blockname, 'installrunning')) {
                require_once($fullblock.'/db/install.php');
                $recover_install_function = 'xmldb_block_'.$blockname.'_install_recovery';
                if (function_exists($recover_install_function)) {
                    $startcallback($component, true, $verbose);
                    $recover_install_function();
                    unset_config('installrunning', 'block_'.$blockname);
                    // Install various components
                    update_capabilities($component);
                    log_update_descriptions($component);
                    external_update_descriptions($component);
                    events_update_definition($component);
                    \core\task\manager::reset_scheduled_tasks_for_component($component);
                    message_update_providers($component);
                    \core\message\inbound\manager::update_handlers_for_component($component);
                    core_tag_area::reset_definitions_for_component($component);
                    $endcallback($component, true, $verbose);
                }
            }
        }

        if (empty($installedversion)) { // block not installed yet, so install it
            $conflictblock = array_search($blocktitle, $blocktitles);
            if ($conflictblock !== false) {
                // Duplicate block titles are not allowed, they confuse people
                // AND PHP's associative arrays ;)
                throw new plugin_defective_exception($component, get_string('blocknameconflict', 'error', (object)array('name'=>$block->name, 'conflict'=>$conflictblock)));
            }
            $startcallback($component, true, $verbose);

            if (file_exists($fullblock.'/db/install.xml')) {
                $DB->get_manager()->install_from_xmldb_file($fullblock.'/db/install.xml');
            }
            $block->id = $DB->insert_record('block', $block);
            upgrade_plugin_savepoint(true, $plugin->version, 'block', $block->name, false);

            if (file_exists($fullblock.'/db/install.php')) {
                require_once($fullblock.'/db/install.php');
                // Set installation running flag, we need to recover after exception or error
                set_config('installrunning', 1, 'block_'.$blockname);
                $post_install_function = 'xmldb_block_'.$blockname.'_install';
                $post_install_function();
                unset_config('installrunning', 'block_'.$blockname);
            }

            $blocktitles[$block->name] = $blocktitle;

            // Install various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            core_tag_area::reset_definitions_for_component($component);

            $endcallback($component, true, $verbose);

        } else if ($installedversion < $plugin->version) {
            $startcallback($component, false, $verbose);

            if (is_readable($fullblock.'/db/upgrade.php')) {
                require_once($fullblock.'/db/upgrade.php');  // defines new upgrading function
                $newupgrade_function = 'xmldb_block_'.$blockname.'_upgrade';
                $result = $newupgrade_function($installedversion, $block);
            } else {
                $result = true;
            }

            $installedversion = $DB->get_field('config_plugins', 'value', array('name'=>'version', 'plugin'=>$component)); // No caching!
            $currblock = $DB->get_record('block', array('name'=>$block->name));
            if ($installedversion < $plugin->version) {
                // store version if not already there
                upgrade_plugin_savepoint($result, $plugin->version, 'block', $block->name, false);
            }

            if ($currblock->cron != $block->cron) {
                // update cron flag if needed
                $DB->set_field('block', 'cron', $block->cron, array('id' => $currblock->id));
            }

            // Upgrade various components
            update_capabilities($component);
            log_update_descriptions($component);
            external_update_descriptions($component);
            events_update_definition($component);
            \core\task\manager::reset_scheduled_tasks_for_component($component);
            message_update_providers($component);
            \core\message\inbound\manager::update_handlers_for_component($component);
            core_tag_area::reset_definitions_for_component($component);

            $endcallback($component, false, $verbose);

        } else if ($installedversion > $plugin->version) {
            throw new downgrade_exception($component, $installedversion, $plugin->version);
        }
    }


    // Finally, if we are in the first_install of BLOCKS setup frontpage and admin page blocks
    if ($first_install) {
        //Iterate over each course - there should be only site course here now
        if ($courses = $DB->get_records('course')) {
            foreach ($courses as $course) {
                blocks_add_default_course_blocks($course);
            }
        }

        blocks_add_default_system_blocks();
    }
}


/**
 * Log_display description function used during install and upgrade.
 *
 * @param string $component name of component (moodle, mod_assignment, etc.)
 * @return void
 */
function log_update_descriptions($component) {
    global $DB;

    $defpath = core_component::get_component_directory($component).'/db/log.php';

    if (!file_exists($defpath)) {
        $DB->delete_records('log_display', array('component'=>$component));
        return;
    }

    // load new info
    $logs = array();
    include($defpath);
    $newlogs = array();
    foreach ($logs as $log) {
        $newlogs[$log['module'].'-'.$log['action']] = $log; // kind of unique name
    }
    unset($logs);
    $logs = $newlogs;

    $fields = array('module', 'action', 'mtable', 'field');
    // update all log fist
    $dblogs = $DB->get_records('log_display', array('component'=>$component));
    foreach ($dblogs as $dblog) {
        $name = $dblog->module.'-'.$dblog->action;

        if (empty($logs[$name])) {
            $DB->delete_records('log_display', array('id'=>$dblog->id));
            continue;
        }

        $log = $logs[$name];
        unset($logs[$name]);

        $update = false;
        foreach ($fields as $field) {
            if ($dblog->$field != $log[$field]) {
                $dblog->$field = $log[$field];
                $update = true;
            }
        }
        if ($update) {
            $DB->update_record('log_display', $dblog);
        }
    }
    foreach ($logs as $log) {
        $dblog = (object)$log;
        $dblog->component = $component;
        $DB->insert_record('log_display', $dblog);
    }
}

/**
 * Web service discovery function used during install and upgrade.
 * @param string $component name of component (moodle, mod_assignment, etc.)
 * @return void
 */
function external_update_descriptions($component) {
    global $DB, $CFG;

    $defpath = core_component::get_component_directory($component).'/db/services.php';

    if (!file_exists($defpath)) {
        require_once($CFG->dirroot.'/lib/externallib.php');
        external_delete_descriptions($component);
        return;
    }

    // load new info
    $functions = array();
    $services = array();
    include($defpath);

    // update all function fist
    $dbfunctions = $DB->get_records('external_functions', array('component'=>$component));
    foreach ($dbfunctions as $dbfunction) {
        if (empty($functions[$dbfunction->name])) {
            $DB->delete_records('external_functions', array('id'=>$dbfunction->id));
            // do not delete functions from external_services_functions, beacuse
            // we want to notify admins when functions used in custom services disappear

            //TODO: this looks wrong, we have to delete it eventually (skodak)
            continue;
        }

        $function = $functions[$dbfunction->name];
        unset($functions[$dbfunction->name]);
        $function['classpath'] = empty($function['classpath']) ? null : $function['classpath'];

        $update = false;
        if ($dbfunction->classname != $function['classname']) {
            $dbfunction->classname = $function['classname'];
            $update = true;
        }
        if ($dbfunction->methodname != $function['methodname']) {
            $dbfunction->methodname = $function['methodname'];
            $update = true;
        }
        if ($dbfunction->classpath != $function['classpath']) {
            $dbfunction->classpath = $function['classpath'];
            $update = true;
        }
        $functioncapabilities = array_key_exists('capabilities', $function)?$function['capabilities']:'';
        if ($dbfunction->capabilities != $functioncapabilities) {
            $dbfunction->capabilities = $functioncapabilities;
            $update = true;
        }

        if (isset($function['services']) and is_array($function['services'])) {
            sort($function['services']);
            $functionservices = implode(',', $function['services']);
        } else {
            // Force null values in the DB.
            $functionservices = null;
        }

        if ($dbfunction->services != $functionservices) {
            // Now, we need to check if services were removed, in that case we need to remove the function from them.
            $servicesremoved = array_diff(explode(",", $dbfunction->services), explode(",", $functionservices));
            foreach ($servicesremoved as $removedshortname) {
                if ($externalserviceid = $DB->get_field('external_services', 'id', array("shortname" => $removedshortname))) {
                    $DB->delete_records('external_services_functions', array('functionname' => $dbfunction->name,
                                                                                'externalserviceid' => $externalserviceid));
                }
            }

            $dbfunction->services = $functionservices;
            $update = true;
        }
        if ($update) {
            $DB->update_record('external_functions', $dbfunction);
        }
    }
    foreach ($functions as $fname => $function) {
        $dbfunction = new stdClass();
        $dbfunction->name       = $fname;
        $dbfunction->classname  = $function['classname'];
        $dbfunction->methodname = $function['methodname'];
        $dbfunction->classpath  = empty($function['classpath']) ? null : $function['classpath'];
        $dbfunction->component  = $component;
        $dbfunction->capabilities = array_key_exists('capabilities', $function)?$function['capabilities']:'';

        if (isset($function['services']) and is_array($function['services'])) {
            sort($function['services']);
            $dbfunction->services = implode(',', $function['services']);
        } else {
            // Force null values in the DB.
            $dbfunction->services = null;
        }

        $dbfunction->id = $DB->insert_record('external_functions', $dbfunction);
    }
    unset($functions);

    // now deal with services
    $dbservices = $DB->get_records('external_services', array('component'=>$component));
    foreach ($dbservices as $dbservice) {
        if (empty($services[$dbservice->name])) {
            $DB->delete_records('external_tokens', array('externalserviceid'=>$dbservice->id));
            $DB->delete_records('external_services_functions', array('externalserviceid'=>$dbservice->id));
            $DB->delete_records('external_services_users', array('externalserviceid'=>$dbservice->id));
            $DB->delete_records('external_services', array('id'=>$dbservice->id));
            continue;
        }
        $service = $services[$dbservice->name];
        unset($services[$dbservice->name]);
        $service['enabled'] = empty($service['enabled']) ? 0 : $service['enabled'];
        $service['requiredcapability'] = empty($service['requiredcapability']) ? null : $service['requiredcapability'];
        $service['restrictedusers'] = !isset($service['restrictedusers']) ? 1 : $service['restrictedusers'];
        $service['downloadfiles'] = !isset($service['downloadfiles']) ? 0 : $service['downloadfiles'];
        $service['uploadfiles'] = !isset($service['uploadfiles']) ? 0 : $service['uploadfiles'];
        $service['shortname'] = !isset($service['shortname']) ? null : $service['shortname'];

        $update = false;
        if ($dbservice->requiredcapability != $service['requiredcapability']) {
            $dbservice->requiredcapability = $service['requiredcapability'];
            $update = true;
        }
        if ($dbservice->restrictedusers != $service['restrictedusers']) {
            $dbservice->restrictedusers = $service['restrictedusers'];
            $update = true;
        }
        if ($dbservice->downloadfiles != $service['downloadfiles']) {
            $dbservice->downloadfiles = $service['downloadfiles'];
            $update = true;
        }
        if ($dbservice->uploadfiles != $service['uploadfiles']) {
            $dbservice->uploadfiles = $service['uploadfiles'];
            $update = true;
        }
        //if shortname is not a PARAM_ALPHANUMEXT, fail (tested here for service update and creation)
        if (isset($service['shortname']) and
                (clean_param($service['shortname'], PARAM_ALPHANUMEXT) != $service['shortname'])) {
            throw new moodle_exception('installserviceshortnameerror', 'webservice', '', $service['shortname']);
        }
        if ($dbservice->shortname != $service['shortname']) {
            //check that shortname is unique
            if (isset($service['shortname'])) { //we currently accepts multiple shortname == null
                $existingservice = $DB->get_record('external_services',
                        array('shortname' => $service['shortname']));
                if (!empty($existingservice)) {
                    throw new moodle_exception('installexistingserviceshortnameerror', 'webservice', '', $service['shortname']);
                }
            }
            $dbservice->shortname = $service['shortname'];
            $update = true;
        }
        if ($update) {
            $DB->update_record('external_services', $dbservice);
        }

        $functions = $DB->get_records('external_services_functions', array('externalserviceid'=>$dbservice->id));
        foreach ($functions as $function) {
            $key = array_search($function->functionname, $service['functions']);
            if ($key === false) {
                $DB->delete_records('external_services_functions', array('id'=>$function->id));
            } else {
                unset($service['functions'][$key]);
            }
        }
        foreach ($service['functions'] as $fname) {
            $newf = new stdClass();
            $newf->externalserviceid = $dbservice->id;
            $newf->functionname      = $fname;
            $DB->insert_record('external_services_functions', $newf);
        }
        unset($functions);
    }
    foreach ($services as $name => $service) {
        //check that shortname is unique
        if (isset($service['shortname'])) { //we currently accepts multiple shortname == null
            $existingservice = $DB->get_record('external_services',
                    array('shortname' => $service['shortname']));
            if (!empty($existingservice)) {
                throw new moodle_exception('installserviceshortnameerror', 'webservice');
            }
        }

        $dbservice = new stdClass();
        $dbservice->name               = $name;
        $dbservice->enabled            = empty($service['enabled']) ? 0 : $service['enabled'];
        $dbservice->requiredcapability = empty($service['requiredcapability']) ? null : $service['requiredcapability'];
        $dbservice->restrictedusers    = !isset($service['restrictedusers']) ? 1 : $service['restrictedusers'];
        $dbservice->downloadfiles      = !isset($service['downloadfiles']) ? 0 : $service['downloadfiles'];
        $dbservice->uploadfiles        = !isset($service['uploadfiles']) ? 0 : $service['uploadfiles'];
        $dbservice->shortname          = !isset($service['shortname']) ? null : $service['shortname'];
        $dbservice->component          = $component;
        $dbservice->timecreated        = time();
        $dbservice->id = $DB->insert_record('external_services', $dbservice);
        foreach ($service['functions'] as $fname) {
            $newf = new stdClass();
            $newf->externalserviceid = $dbservice->id;
            $newf->functionname      = $fname;
            $DB->insert_record('external_services_functions', $newf);
        }
    }
}

/**
 * Allow plugins and subsystems to add external functions to other plugins or built-in services.
 * This function is executed just after all the plugins have been updated.
 */
function external_update_services() {
    global $DB;

    // Look for external functions that want to be added in existing services.
    $functions = $DB->get_records_select('external_functions', 'services IS NOT NULL');

    $servicescache = array();
    foreach ($functions as $function) {
        // Prevent edge cases.
        if (empty($function->services)) {
            continue;
        }
        $services = explode(',', $function->services);

        foreach ($services as $serviceshortname) {
            if ($serviceshortname === 'moodle_mobile_app') {
                // Totara: not compatible with Totara.
                continue;
            }
            // Get the service id by shortname.
            if (!empty($servicescache[$serviceshortname])) {
                $serviceid = $servicescache[$serviceshortname];
            } else if ($service = $DB->get_record('external_services', array('shortname' => $serviceshortname))) {
                // If the component is empty, it means that is not a built-in service.
                // We don't allow functions to inject themselves in services created by an user in Moodle.
                if (empty($service->component)) {
                    continue;
                }
                $serviceid = $service->id;
                $servicescache[$serviceshortname] = $serviceid;
            } else {
                // Service not found.
                continue;
            }
            // Finally add the function to the service.
            $newf = new stdClass();
            $newf->externalserviceid = $serviceid;
            $newf->functionname      = $function->name;

            if (!$DB->record_exists('external_services_functions', (array)$newf)) {
                $DB->insert_record('external_services_functions', $newf);
            }
        }
    }
}

/**
 * upgrade logging functions
 */
function upgrade_handle_exception($ex, $plugin = null) {
    global $CFG;

    // rollback everything, we need to log all upgrade problems
    abort_all_db_transactions();

    $info = get_exception_info($ex);

    // First log upgrade error
    upgrade_log(UPGRADE_LOG_ERROR, $plugin, 'Exception: ' . get_class($ex), $info->message, $info->backtrace);

    // Always turn on debugging - admins need to know what is going on
    set_debugging(DEBUG_DEVELOPER, true);

    default_exception_handler($ex, true, $plugin);
}

/**
 * Adds log entry into upgrade_log table
 *
 * @param int $type UPGRADE_LOG_NORMAL, UPGRADE_LOG_NOTICE or UPGRADE_LOG_ERROR
 * @param string $plugin frankenstyle component name
 * @param string $info short description text of log entry
 * @param string $details long problem description
 * @param string $backtrace string used for errors only
 * @return void
 */
function upgrade_log($type, $plugin, $info, $details=null, $backtrace=null) {
    global $DB, $USER, $CFG;

    if (empty($plugin)) {
        $plugin = 'core';
    }

    list($plugintype, $pluginname) = core_component::normalize_component($plugin);
    $component = is_null($pluginname) ? $plugintype : $plugintype . '_' . $pluginname;

    $backtrace = format_backtrace($backtrace, true);

    $currentversion = null;
    $targetversion  = null;

    //first try to find out current version number
    if ($plugintype === 'core') {
        //main
        $currentversion = $CFG->version;

        $version = null;
        include("$CFG->dirroot/version.php");
        $targetversion = $version;

    } else {
        $pluginversion = get_config($component, 'version');
        if (!empty($pluginversion)) {
            $currentversion = $pluginversion;
        }
        $cd = core_component::get_component_directory($component);
        if (file_exists("$cd/version.php")) {
            $plugin = new stdClass();
            $plugin->version = null;
            $module = $plugin;
            include("$cd/version.php");
            $targetversion = $plugin->version;
        }
    }

    $log = new stdClass();
    $log->type          = $type;
    $log->plugin        = $component;
    $log->version       = $currentversion;
    $log->targetversion = $targetversion;
    $log->info          = $info;
    $log->details       = $details;
    $log->backtrace     = $backtrace;
    $log->userid        = $USER->id;
    $log->timemodified  = time();
    try {
        $DB->insert_record('upgrade_log', $log);
    } catch (Exception $ignored) {
        // possible during install or 2.0 upgrade
    }
}

/**
 * Marks start of upgrade, blocks any other access to site.
 * The upgrade is finished at the end of script or after timeout.
 *
 * @global object
 * @global object
 * @global object
 */
function upgrade_started($preinstall=false) {
    global $CFG, $DB, $PAGE, $OUTPUT, $TOTARA;

    static $started = false;

    if ($preinstall) {
        ignore_user_abort(true);
        upgrade_setup_debug(true);

    } else if ($started) {
        upgrade_set_timeout(120);

    } else {
        if (!CLI_SCRIPT and !$PAGE->headerprinted) {
            $strupgrade  = get_string('upgradingversion', 'admin');
            $PAGE->set_pagelayout('maintenance');
            upgrade_init_javascript();
            $PAGE->set_title($strupgrade.' - Totara '.$TOTARA->release);
            $PAGE->set_heading($strupgrade);
            $PAGE->navbar->add($strupgrade);
            $PAGE->set_cacheable(false);
            echo $OUTPUT->header();
        }

        ignore_user_abort(true);
        core_shutdown_manager::register_function('upgrade_finished_handler');
        upgrade_setup_debug(true);
        set_config('upgraderunning', time()+300);
        $started = true;
    }
}

/**
 * Internal function - executed if upgrade interrupted.
 */
function upgrade_finished_handler() {
    upgrade_finished();
}

/**
 * Indicates upgrade is finished.
 *
 * This function may be called repeatedly.
 *
 * @global object
 * @global object
 */
function upgrade_finished($continueurl=null) {
    global $CFG, $DB, $OUTPUT;

    if (!empty($CFG->upgraderunning)) {
        unset_config('upgraderunning');
        // We have to forcefully purge the caches using the writer here.
        // This has to be done after we unset the config var. If someone hits the site while this is set they will
        // cause the config values to propogate to the caches.
        // Caches are purged after the last step in an upgrade but there is several code routines that exceute between
        // then and now that leaving a window for things to fall out of sync.
        cache_helper::purge_all(true);
        upgrade_setup_debug(false);
        ignore_user_abort(false);
        if ($continueurl) {
            echo $OUTPUT->continue_button($continueurl);
            echo $OUTPUT->footer();
            die;
        }
    }
}

/**
 * @global object
 * @global object
 */
function upgrade_setup_debug($starting) {
    global $CFG, $DB;

    static $originaldebug = null;

    if ($starting) {
        if ($originaldebug === null) {
            $originaldebug = $DB->get_debug();
        }
        if (!empty($CFG->upgradeshowsql)) {
            $DB->set_debug(true);
        }
    } else {
        $DB->set_debug($originaldebug);
    }
}

function print_upgrade_separator() {
    if (!CLI_SCRIPT) {
        echo '<hr />';
    }
}

/**
 * Default start upgrade callback
 * @param string $plugin
 * @param bool $installation true if installation, false means upgrade
 */
function print_upgrade_part_start($plugin, $installation, $verbose) {
    global $OUTPUT;
    if (empty($plugin) or $plugin == 'moodle') {
        upgrade_started($installation); // does not store upgrade running flag yet
        if ($verbose) {
            if (empty($plugin)) {
                echo $OUTPUT->heading('Totara');
            } else {
                echo $OUTPUT->heading(get_string('coresystem'));
            }
        }
    } else {
        upgrade_started();
        if ($verbose) {
            echo $OUTPUT->heading($plugin);
        }
    }
    if ($installation) {
        if (empty($plugin) or $plugin == 'moodle') {
            // no need to log - log table not yet there ;-)
        } else {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Starting plugin installation', get_upgrade_system_info());
        }
    } else {
        core_upgrade_time::record_start();
        if (empty($plugin) or $plugin == 'moodle') {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Starting core upgrade', get_upgrade_system_info());
        } else {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Starting plugin upgrade', get_upgrade_system_info());
        }
    }
}

/**
 * Returns basic system information for upgrade logging purposes.
 *
 * @internal
 *
 * @return string
 */
function get_upgrade_system_info() {
    global $CFG, $DB;

    $phpversion = phpversion();
    $dbtype = $CFG->dbtype;
    $dbversion = $DB->get_server_info()['version'];
    $osdetails = php_uname('s') . " " . php_uname('r') . " " . php_uname('m');

    return "PHP: $phpversion, $dbtype: $dbversion, OS: $osdetails";
}

/**
 * Default end upgrade callback
 * @param string $plugin
 * @param bool $installation true if installation, false means upgrade
 */
function print_upgrade_part_end($plugin, $installation, $verbose) {
    global $OUTPUT;
    upgrade_started();
    if ($installation) {
        if (empty($plugin) or $plugin == 'moodle') {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Core installed');
        } else {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Plugin installed');
        }
    } else {
        if (empty($plugin) or $plugin == 'moodle') {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Core upgraded');
        } else {
            upgrade_log(UPGRADE_LOG_NORMAL, $plugin, 'Plugin upgraded');
        }
    }
    if ($verbose) {
        if ($installation) {
            $message = get_string('success');
        } else {
            $duration = core_upgrade_time::get_elapsed();
            $message = get_string('successduration', '', format_float($duration, 2));
        }
        $notification = new \core\output\notification($message, \core\output\notification::NOTIFY_SUCCESS);
        $notification->set_show_closebutton(false);
        echo $OUTPUT->render($notification);
        print_upgrade_separator();
    }
}

/**
 * Sets up JS code required for all upgrade scripts.
 * @global object
 */
function upgrade_init_javascript() {
    global $PAGE;
    // scroll to the end of each upgrade page so that ppl see either error or continue button,
    // no need to scroll continuously any more, it is enough to jump to end once the footer is printed ;-)
    $js = "window.scrollTo(0, 5000000);";
    $PAGE->requires->js_init_code($js);
}

/**
 * Try to upgrade the given language pack (or current language)
 *
 * @param string $lang the code of the language to update, defaults to the current language
 */
function upgrade_language_pack($lang = null) {
    global $CFG;

    if (!empty($CFG->skiplangupgrade)) {
        return;
    }

    if (!file_exists("$CFG->dirroot/$CFG->admin/tool/langimport/lib.php")) {
        // weird, somebody uninstalled the import utility
        return;
    }

    if (!$lang) {
        $lang = current_language();
    }

    if (!get_string_manager()->translation_exists($lang)) {
        return;
    }

    get_string_manager()->reset_caches();

    if ($lang === 'en') {
        return;  // Nothing to do
    }

    upgrade_started(false);

    require_once("$CFG->dirroot/$CFG->admin/tool/langimport/lib.php");
    tool_langimport_preupgrade_update($lang);

    get_string_manager()->reset_caches();

    print_upgrade_separator();
}

/**
 * Install core moodle tables and initialize
 * @param float $version target version
 * @param bool $verbose
 * @return void, may throw exception
 */
function install_core($version, $verbose) {
    global $CFG, $DB;

    // Totara: Make sure we have the correct versions, ignore the sloppy param.
    $version = null;
    $release = null;
    $branch = null;
    $TOTARA = new stdClass();
    require("$CFG->dirroot/version.php");

    // We can not call purge_all_caches() yet, make sure the temp and cache dirs exist and are empty.
    remove_dir($CFG->cachedir.'', true);
    make_cache_directory('', true);

    remove_dir($CFG->localcachedir.'', true);
    make_localcache_directory('', true);

    remove_dir($CFG->tempdir.'', true);
    make_temp_directory('', true);

    remove_dir($CFG->dataroot.'/muc', true);
    make_writable_directory($CFG->dataroot.'/muc', true);

    try {
        core_php_time_limit::raise(600);
        print_upgrade_part_start('moodle', true, $verbose); // does not store upgrade running flag

        $DB->get_manager()->install_from_xmldb_file("$CFG->libdir/db/install.xml");
        upgrade_started();     // we want the flag to be stored in config table ;-)

        // set all core default records and default settings
        require_once("$CFG->libdir/db/install.php");
        xmldb_main_install(); // installs the capabilities too

        // store version
        upgrade_main_savepoint(true, $version, false);

        // Continue with the installation
        log_update_descriptions('moodle');
        external_update_descriptions('moodle');
        events_update_definition('moodle');
        \core\task\manager::reset_scheduled_tasks_for_component('moodle');
        message_update_providers('moodle');
        \core\message\inbound\manager::update_handlers_for_component('moodle');
        core_tag_area::reset_definitions_for_component('moodle');

        // Write default settings unconditionally
        admin_apply_default_settings(NULL, true);

        // Totara: update all version info consistently.
        set_config('release', $release);
        set_config('branch', $branch);
        set_config('totara_release', $TOTARA->release);
        set_config('totara_build', $TOTARA->build);
        set_config('totara_version', $TOTARA->version);

        // If there is any forced flavour activate it and tell it that it was just activated,
        // flavours might want to hack some settings or do other weird magic.
        \totara_flavour\helper::execute_post_install_steps();

        print_upgrade_part_end(null, true, $verbose);

        // Purge all caches. They're disabled but this ensures that we don't have any persistent data just in case something
        // during installation didn't use APIs.
        cache_helper::purge_all();
    } catch (exception $ex) {
        upgrade_handle_exception($ex);
    } catch (Throwable $ex) {
        // Engine errors in PHP7 throw exceptions of type Throwable (this "catch" will be ignored in PHP5).
        upgrade_handle_exception($ex);
    }
}

/**
 * Upgrade moodle core
 * @param float $version target version
 * @param bool $verbose
 * @return void, may throw exception
 */
function upgrade_core($version, $verbose) {
    global $CFG, $SITE, $DB, $COURSE, $TOTARA;

    // Totara: Make sure we have the correct versions, ignore the sloppy param.
    $version = null;
    $release = null;
    $branch = null;
    $TOTARA = new stdClass();
    require("$CFG->dirroot/version.php");

    raise_memory_limit(MEMORY_EXTRA);

    require_once($CFG->libdir.'/db/upgrade.php');    // Defines upgrades
    require_once($CFG->dirroot.'/totara/core/db/utils.php'); // Include Totara upgrade stuff.

    try {
        // Reset caches before any output.
        cache_helper::purge_all(true);
        purge_all_caches();

        // Totara; Run any pre-upgrade special fixes that may be required and update all languages.
        totara_preupgrade();

        print_upgrade_part_start('moodle', false, $verbose);

        // Pre-upgrade scripts for local hack workarounds.
        $preupgradefile = "$CFG->dirroot/local/preupgrade.php";
        if (file_exists($preupgradefile)) {
            core_php_time_limit::raise();
            require($preupgradefile);
            // Reset upgrade timeout to default.
            upgrade_set_timeout();
        }

        $result = xmldb_main_upgrade($CFG->version);
        if ($version > $CFG->version) {
            // store version if not already there
            upgrade_main_savepoint($result, $version, false);
        }

        // In case structure of 'course' table has been changed and we forgot to update $SITE, re-read it from db.
        $SITE = $DB->get_record('course', array('id' => $SITE->id));
        $COURSE = clone($SITE);

        // perform all other component upgrade routines
        update_capabilities('moodle');
        log_update_descriptions('moodle');
        external_update_descriptions('moodle');
        events_update_definition('moodle');
        \core\task\manager::reset_scheduled_tasks_for_component('moodle');
        message_update_providers('moodle');
        \core\message\inbound\manager::update_handlers_for_component('moodle');
        core_tag_area::reset_definitions_for_component('moodle');
        // Update core definitions.
        cache_helper::update_definitions(true);

        // Purge caches again, just to be sure we arn't holding onto old stuff now.
        cache_helper::purge_all(true);
        purge_all_caches();

        // Clean up contexts - more and more stuff depends on existence of paths and contexts
        context_helper::cleanup_instances();
        context_helper::create_instances(null, false);
        context_helper::build_all_paths(false);
        $syscontext = context_system::instance();
        $syscontext->mark_dirty();

        // Add default encryption key
        core\cipher\key\manager::instance()->add_default_key();

        // Totara: update all version info consistently.
        set_config('release', $release);
        set_config('branch', $branch);
        set_config('totara_release', $TOTARA->release);
        set_config('totara_build', $TOTARA->build);
        set_config('totara_version', $TOTARA->version);

        print_upgrade_part_end('moodle', false, $verbose);
    } catch (Exception $ex) {
        upgrade_handle_exception($ex);
    } catch (Throwable $ex) {
        // Engine errors in PHP7 throw exceptions of type Throwable (this "catch" will be ignored in PHP5).
        upgrade_handle_exception($ex);
    }
}

/**
 * Upgrade/install other parts of moodle
 * @param bool $verbose
 * @return void, may throw exception
 */
function upgrade_noncore($verbose) {
    global $CFG;

    raise_memory_limit(MEMORY_EXTRA);

    require_once($CFG->dirroot.'/totara/core/db/utils.php'); // Include Totara upgrade stuff.

    // upgrade all plugins types
    try {
        // Reset caches before any output.
        cache_helper::purge_all(true);
        purge_all_caches();

        // Totara; Run any pre-upgrade special fixes that may be required.
        totara_preupgrade();

        $plugintypes = core_component::get_plugin_types();

        // Totara: we give totara_core preferential treatment, it always gets executed first.
        $plugintypes = array_merge(array('totaracoreonly' => null), $plugintypes);

        foreach ($plugintypes as $type=>$location) {
            upgrade_plugins($type, 'print_upgrade_part_start', 'print_upgrade_part_end', $verbose);
        }
        // Upgrade services.
        // This function gives plugins and subsystems a chance to add functions to existing built-in services.
        external_update_services();

        // Update cache definitions. Involves scanning each plugin for any changes.
        cache_helper::update_definitions();
        // Mark the site as upgraded.
        set_config('allversionshash', core_component::get_all_versions_hash());

        // Upgrade totara navigation menu.
        totara_upgrade_menu();
        // Allow the Totara Flavour to act upon the upgrade if need be.
        \totara_flavour\helper::execute_post_upgrade_steps();

        // Purge caches again, just to be sure we arn't holding onto old stuff now.
        cache_helper::purge_all(true);
        purge_all_caches();

    } catch (Exception $ex) {
        upgrade_handle_exception($ex);
    } catch (Throwable $ex) {
        // Engine errors in PHP7 throw exceptions of type Throwable (this "catch" will be ignored in PHP5).
        upgrade_handle_exception($ex);
    }
}

/**
 * Checks if the main tables have been installed yet or not.
 *
 * Note: we can not use caches here because they might be stale,
 *       use with care!
 *
 * @return bool
 */
function core_tables_exist() {
    global $DB;

    if (!$tables = $DB->get_tables(false) ) {    // No tables yet at all.
        return false;

    } else {                                 // Check for missing main tables
        $mtables = array('config', 'course', 'groupings'); // some tables used in 1.9 and 2.0, preferable something from the start and end of install.xml
        foreach ($mtables as $mtable) {
            if (!in_array($mtable, $tables)) {
                return false;
            }
        }
        return true;
    }
}

/**
 * Detect draft file areas with missing root directory records and add them.
 */
function upgrade_fix_missing_root_folders_draft() {
    global $DB;

    $transaction = $DB->start_delegated_transaction();

    $sql = "SELECT contextid, itemid, MAX(timecreated) AS timecreated, MAX(timemodified) AS timemodified
              FROM {files}
             WHERE (component = 'user' AND filearea = 'draft')
          GROUP BY contextid, itemid
            HAVING MAX(CASE WHEN filename = '.' AND filepath = '/' THEN 1 ELSE 0 END) = 0";

    $rs = $DB->get_recordset_sql($sql);
    $defaults = array('component' => 'user',
        'filearea' => 'draft',
        'filepath' => '/',
        'filename' => '.',
        'userid' => 0, // Don't rely on any particular user for these system records.
        'filesize' => 0,
        'contenthash' => sha1(''));
    foreach ($rs as $r) {
        $r->pathnamehash = sha1("/$r->contextid/user/draft/$r->itemid/.");
        $DB->insert_record('files', (array)$r + $defaults);
    }
    $rs->close();
    $transaction->allow_commit();
}

/**
 * This function verifies that the database is not using an unsupported storage engine.
 *
 * @param environment_results $result object to update, if relevant
 * @return environment_results|null updated results object, or null if the storage engine is supported
 */
function check_database_storage_engine(environment_results $result) {
    global $DB;

    // Check if MySQL is the DB family (this will also be the same for MariaDB).
    if ($DB->get_dbfamily() == 'mysql') {
        // Get the database engine we will either be using to install the tables, or what we are currently using.
        $engine = $DB->get_dbengine();
        // Check if MyISAM is the storage engine that will be used, if so, do not proceed and display an error.
        if ($engine == 'MyISAM') {
            $result->setInfo(get_string('unsupporteddbstorageenginecheck', 'admin'));
            $result->setStatus(false);
            return $result;
        }
    }

    return null;
}

/**
 * This function verifies if the database has tables using innoDB Antelope row format.
 *
 * @param environment_results $result
 * @return environment_results|null updated results object, or null if no Antelope table has been found.
 */
function check_database_tables_row_format(environment_results $result) {
    global $DB;

    if ($DB->get_dbfamily() === 'mysql') {
        /** @var mysql_sql_generator $generator */
        $generator = $DB->get_manager()->generator;

        foreach ($DB->get_tables(false) as $table) {
            $columns = $DB->get_columns($table, false);
            $size = $generator->guess_antelope_row_size($columns);
            $format = $DB->get_row_format($table);

            if ($format === 'Compressed') {
                continue;

            } else if ($format === 'Dynamic') {
                if ($size > $generator::ANTELOPE_MAX_ROW_SIZE) {
                    $result->setInfo(get_string('unsupporteddbtablerowformatcheck', 'admin'));
                    $result->setStatus(false);
                    return $result;
                }

            } else {
                $result->setInfo(get_string('unsupporteddbtablerowformatcheck', 'admin'));
                $result->setStatus(false);
                return $result;
            }
        }
    }

    return null;
}

/**
 * Totara: This function verifies that the database uses Barracuda file format.
 *
 * @param environment_results $result
 * @return environment_results|null updated results object, or null
 */
function check_mysql_file_format(environment_results $result) {
    global $DB;

    if ($DB->get_dbfamily() !== 'mysql') {
        return null;
    }

    /** @var mysqli_native_moodle_database $DB */

    if ($DB->get_file_format() !== "Barracuda") {
        $result->setStatus(false);
        return $result;
    }
    $result->setStatus(true);
    return $result;
}

/**
 * Totara: This function verifies that the database has a setting of one file per table.
 *
 * @param environment_results $result
 * @return environment_results|null updated results object, or null
 */
function check_mysql_file_per_table(environment_results $result) {
    global $DB;

    if ($DB->get_dbfamily() !== 'mysql') {
        return null;
    }

    if (!$DB->is_file_per_table_enabled()) {
        $result->setStatus(false);
        return $result;
    }
    $result->setStatus(true);
    return $result;
}

/**
 * Totara: This function verifies that the database has the setting of large prefix enabled. This is required only for 'utf8mb4'.
 *
 * @param environment_results $result
 * @return environment_results|null updated results object, or null
 */
function check_mysql_large_prefix(environment_results $result) {
    global $DB;

    if ($DB->get_dbfamily() === 'mysql') {
        $charset = $DB->get_charset();
        if ($charset === 'utf8mb4') {
            if (!$DB->is_large_prefix_enabled()) {
                $result->setFeedbackStr('unsupporteddblargeprefix');
                $result->setStatus(false);
                return $result;
            }
            $result->setStatus(true);
            return $result;
        } else {
            // Totara: we want large prefix for Report builder caching!
            $result->setLevel('optional');
            if (!$DB->is_large_prefix_enabled()) {
                $result->setFeedbackStr('unsupporteddblargeprefix');
                $result->setStatus(false);
                return $result;
            }
            $result->setStatus(true);
            return $result;
        }
    }
    return null;
}

/**
 * Check if the site is being served using an ssl url.
 *
 * Note this does not really perform any request neither looks for proxies or
 * other situations. Just looks to wwwroot and warn if it's not using https.
 *
 * @param  environment_results $result $result
 * @return environment_results|null updated results object, or null if the site is https.
 */
function check_is_https(environment_results $result) {
    global $CFG;

    // Only if is defined, non-empty and whatever core tell us.
    if (!empty($CFG->wwwroot) && !is_https()) {
        $result->setInfo(get_string('sitenothttps', 'admin'));
        $result->setStatus(false);
        return $result;
    }
    return null;
}

/**
 * Upgrade the minmaxgrade setting.
 *
 * This step should only be run for sites running 2.8 or later. Sites using 2.7 will be fine
 * using the new default system setting $CFG->grade_minmaxtouse.
 *
 * @return void
 */
function upgrade_minmaxgrade() {
    global $CFG, $DB;

    // 2 is a copy of GRADE_MIN_MAX_FROM_GRADE_GRADE.
    $settingvalue = 2;

    // Set the course setting when:
    // - The system setting does not exist yet.
    // - The system seeting is not set to what we'd set the course setting.
    $setcoursesetting = !isset($CFG->grade_minmaxtouse) || $CFG->grade_minmaxtouse != $settingvalue;

    // Identify the courses that have inconsistencies grade_item vs grade_grade.
    $sql = "SELECT DISTINCT(gi.courseid)
              FROM {grade_grades} gg
              JOIN {grade_items} gi
                ON gg.itemid = gi.id
             WHERE gi.itemtype NOT IN (?, ?)
               AND (gg.rawgrademax != gi.grademax OR gg.rawgrademin != gi.grademin)";

    $rs = $DB->get_recordset_sql($sql, array('course', 'category'));
    foreach ($rs as $record) {
        // Flag the course to show a notice in the gradebook.
        set_config('show_min_max_grades_changed_' . $record->courseid, 1);

        // Set the appropriate course setting so that grades displayed are not changed.
        $configname = 'minmaxtouse';
        if ($setcoursesetting &&
                !$DB->record_exists('grade_settings', array('courseid' => $record->courseid, 'name' => $configname))) {
            // Do not set the setting when the course already defines it.
            $data = new stdClass();
            $data->courseid = $record->courseid;
            $data->name     = $configname;
            $data->value    = $settingvalue;
            $DB->insert_record('grade_settings', $data);
        }

        // Mark the grades to be regraded.
        $DB->set_field('grade_items', 'needsupdate', 1, array('courseid' => $record->courseid));
    }
    $rs->close();
}


/**
 * Assert the upgrade key is provided, if it is defined.
 *
 * The upgrade key can be defined in the main config.php as $CFG->upgradekey. If
 * it is defined there, then its value must be provided every time the site is
 * being upgraded, regardless the administrator is logged in or not.
 *
 * This is supposed to be used at certain places in /admin/index.php only.
 *
 * @param string|null $upgradekeyhash the SHA-1 of the value provided by the user
 */
function check_upgrade_key($upgradekeyhash) {
    global $CFG, $PAGE;

    if (isset($CFG->config_php_settings['upgradekey'])) {
        if ($upgradekeyhash === null or $upgradekeyhash !== sha1($CFG->config_php_settings['upgradekey'])) {
            if (!$PAGE->headerprinted) {
                $output = $PAGE->get_renderer('core', 'admin');
                echo $output->upgradekey_form_page(new moodle_url('/admin/index.php', array('cache' => 0)));
                die();
            } else {
                // This should not happen.
                die('Upgrade locked');
            }
        }
    } else {
        $output = $PAGE->get_renderer('core', 'admin');
        echo $output->upgradekey_warning_from_page();
        exit();
    }
}

/**
 * Method used to check the installed unoconv version.
 *
 * @param environment_results $result object to update, if relevant.
 * @return environment_results|null updated results or null if unoconv path is not executable.
 */
function check_unoconv_version(environment_results $result) {
    global $CFG;

    if (!during_initial_install() && !empty($CFG->pathtounoconv)) {
        require_once($CFG->libdir . '/filelib.php');
        if (file_is_executable(trim($CFG->pathtounoconv))) {
            $currentversion = 0;
            $supportedversion = 0.7;
            $unoconvbin = \escapeshellarg($CFG->pathtounoconv);
            $command = "$unoconvbin --version";
            exec($command, $output);

            // If the command execution returned some output, then get the unoconv version.
            if ($output) {
                foreach ($output as $response) {
                    if (preg_match('/unoconv (\\d+\\.\\d+)/', $response, $matches)) {
                        $currentversion = (float)$matches[1];
                    }
                }
            }

            if ($currentversion < $supportedversion) {
                $result->setInfo(get_string('unsupportedunoconvversion', 'admin'));
                $result->setStatus(false);
                return $result;
            }
        }
    }
    return null;
}

/**
 * Checks for up-to-date TLS libraries. NOTE: this is not currently used, see MDL-57262.
 *
 * @param environment_results $result object to update, if relevant.
 * @return environment_results|null updated results or null if unoconv path is not executable.
 */
function check_tls_libraries(environment_results $result) {
    global $CFG;

    if (!function_exists('curl_version')) {
        $result->setInfo(get_string('curlphpextensionnotinstalled', 'admin'));
        $result->setStatus(false);
        return $result;
    }

    if (!\core\upgrade\util::validate_php_curl_tls(curl_version(), PHP_ZTS)) {
        $result->setInfo(get_string('invalidssltlsconfiguration', 'admin'));
        $result->setStatus(false);
        return $result;
    }

    if (!\core\upgrade\util::can_use_tls12(curl_version(), php_uname('r'))) {
        $result->setInfo(get_string('unsupportedssltlsconfiguration', 'admin'));
        $result->setStatus(false);
        return $result;
    }

    return null;
}

/**
 * Check if recommended version of libcurl is installed or not.
 *
 * @param environment_results $result object to update, if relevant.
 * @return environment_results|null updated results or null.
 */
function check_libcurl_version(environment_results $result) {

    if (!function_exists('curl_version')) {
        $result->setInfo(get_string('curlphpextensionnotinstalled', 'admin'));
        $result->setStatus(false);
        return $result;
    }

    // Supported version and version number.
    $supportedversion = 0x071304;
    $supportedversionstring = "7.19.4";

    // Installed version.
    $curlinfo = curl_version();
    $currentversion = $curlinfo['version_number'];

    if ($currentversion < $supportedversion) {
        // Test fail.
        // Set info, we want to let user know how to resolve the problem.
        $result->setInfo('Libcurl version check');
        $result->setNeededVersion($supportedversionstring);
        $result->setCurrentVersion($curlinfo['version']);
        $result->setStatus(false);
        return $result;
    }

    return null;
}

/**
 * Fix how auth plugins are called in the 'config_plugins' table.
 *
 * For legacy reasons, the auth plugins did not always use their frankenstyle
 * component name in the 'plugin' column of the 'config_plugins' table. This is
 * a helper function to correctly migrate the legacy settings into the expected
 * and consistent way.
 *
 * @param string $plugin the auth plugin name such as 'cas', 'manual'
 */
function upgrade_fix_config_auth_plugin_names($plugin) {
    global $CFG, $DB, $OUTPUT;

    $legacy = (array) get_config('auth/'.$plugin);
    $current = (array) get_config('auth_'.$plugin);

    // I don't want to rely on array_merge() and friends here just in case
    // there was some crazy setting with a numerical name.

    if ($legacy) {
        $new = $legacy;
    } else {
        $new = [];
    }

    if ($current) {
        foreach ($current as $name => $value) {
            if (isset($legacy[$name]) && ($legacy[$name] !== $value)) {
                // No need to pollute the output during unit tests.
                if (!empty($CFG->upgraderunning)) {
                    $message = get_string('settingmigrationmismatch', 'core_auth', [
                        'plugin' => 'auth_'.$plugin,
                        'setting' => s($name),
                        'legacy' => s($legacy[$name]),
                        'current' => s($value),
                    ]);
                    echo $OUTPUT->notification($message, \core\output\notification::NOTIFY_ERROR);

                    upgrade_log(UPGRADE_LOG_NOTICE, 'auth_'.$plugin, 'Setting values mismatch detected',
                        'SETTING: '.$name. ' LEGACY: '.$legacy[$name].' CURRENT: '.$value);
                }
            }

            $new[$name] = $value;
        }
    }

    foreach ($new as $name => $value) {
        set_config($name, $value, 'auth_'.$plugin);
        unset_config($name, 'auth/'.$plugin);
    }
}

/**
 * Populate the auth plugin settings with defaults if needed.
 *
 * As a result of fixing the auth plugins config storage, many settings would
 * be falsely reported as new ones by admin/upgradesettings.php. We do not want
 * to confuse admins so we try to reduce the bewilderment by pre-populating the
 * config_plugins table with default values. This should be done only for
 * disabled auth methods. The enabled methods have their settings already
 * stored, so reporting actual new settings for them is valid.
 *
 * @deprecated since Totara 12.0
 *
 * @param string $plugin the auth plugin name such as 'cas', 'manual'
 */
function upgrade_fix_config_auth_plugin_defaults($plugin) {
    // Totara: it would be very naive to abuse settings.php files here, it would NOT work!
    //         Instead set any missing auth plugin default settings in upgrade.php directly.
    debugging('Missing auth plugin defaults cannot be added automatically');
}

/**
 * Search for a given theme in any of the parent themes of a given theme.
 *
 * @param string $needle The name of the theme you want to search for
 * @param string $themename The name of the theme you want to search for
 * @param string $checkedthemeforparents The name of all the themes already checked
 * @return bool True if found, false if not.
 */
function upgrade_theme_is_from_family($needle, $themename, $checkedthemeforparents = []) {
    global $CFG;

    // Once we've started checking a theme, don't start checking it again. Prevent recursion.
    if (!empty($checkedthemeforparents[$themename])) {
        return false;
    }
    $checkedthemeforparents[$themename] = true;

    if ($themename == $needle) {
        return true;
    }

    if ($themedir = upgrade_find_theme_location($themename)) {
        $THEME = new stdClass();
        require($themedir . '/config.php');
        $theme = $THEME;
    } else {
        return false;
    }

    if (empty($theme->parents)) {
        return false;
    }

    // Recursively search through each parent theme.
    foreach ($theme->parents as $parent) {
        if (upgrade_theme_is_from_family($needle, $parent, $checkedthemeforparents)) {
            return true;
        }
    }
    return false;
}

/**
 * Finds the theme location and verifies the theme has all needed files.
 *
 * @param string $themename The name of the theme you want to search for
 * @return string full dir path or null if not found
 * @see \theme_config::find_theme_location()
 */
function upgrade_find_theme_location($themename) {
    global $CFG;

    if (file_exists("$CFG->dirroot/theme/$themename/config.php")) {
        $dir = "$CFG->dirroot/theme/$themename";
    } else if (!empty($CFG->themedir) and file_exists("$CFG->themedir/$themename/config.php")) {
        $dir = "$CFG->themedir/$themename";
    } else {
        return null;
    }

    return $dir;
}