<?php
/**
 * This file is part of Totara Talent Experience Platform
 *
 * Copyright (C) 2023 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Simon Chester <simon.chester@totara.com>
 * @package core_mfa
 */

use core\webapi\resolver\payload;
use core\webapi\resolver\result;
use core_mfa\webapi\middleware\require_can_register;
use core_phpunit\testcase;

/**
 * @coversDefaultClass \core_mfa\webapi\middleware\require_can_register
 * @group core_mfa
 */
class core_mfa_webapi_middleware_require_can_register_test extends testcase {

    protected require_can_register $middleware;
    protected result $result;

    protected function setUp(): void {
        parent::setUp();
        set_config('mfa_plugins', 'button,totp');
        $this->middleware = new require_can_register();
        $this->result = new result(['success' => true]);
    }

    protected function tearDown(): void {
        unset_config('mfa_plugins');
        unset($this->middleware, $this->result);
        parent::tearDown();
    }

    /**
     * Assert that you can register the MFA as an admin
     *
     * @return void
     */
    public function test_with_permission_admin(): void {
        $this->setAdminUser();
        $this->expect_successful_mfa_registration();
    }

    /**
     * Assert that you can register the MFA as a regular user
     *
     * @return void
     */
    public function test_with_permission_regular_user(): void {
        $this->setUser(static::getDataGenerator()->create_user());
        $this->expect_successful_mfa_registration();
    }

    /**
     * Assert that you cannot register the MFA as a guest user
     *
     * @return void
     */
    public function test_without_permission_guest(): void {
        $this->setGuestUser();
        $this->expect_unsuccessful_mfa_registration();
    }

    /**
     * Assert that you cannot register the MFA as a loggedinas user
     *
     * @return void
     */
    public function test_without_permission_loggedinas(): void {
        $this->setAdminUser();
        $user = static::getDataGenerator()->create_user();
        \core\session\manager::loginas($user->id, context_system::instance());
        $this->expect_unsuccessful_mfa_registration();
    }

    /**
     * Expect permission success
     *
     * @return void
     */
    private function expect_successful_mfa_registration(): void {
        $payload = $this->createMock(payload::class);
        $returned = $this->middleware->handle($payload, fn() => $this->result);
        $this->assertSame($this->result, $returned);
    }

    /**
     * Expect permission failure
     *
     * @return void
     */
    private function expect_unsuccessful_mfa_registration(): void {
        $this->expectException(\moodle_exception::class);
        $this->expectExceptionMessage(
            get_string('nopermissions', 'error', get_string('register_a_new_factor', 'mfa'))
        );
        $payload = $this->createMock(payload::class);
        $this->middleware->handle($payload, fn() => $this->result);
    }
}
