<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2021 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Tatsuhiro Kirihara <tatsuhiro.kirihara@totaralearning.com>
 * @package mod_approval
 */

namespace mod_approval\webapi\resolver\mutation;

use core\entity\user;
use core\webapi\execution_context;
use core\webapi\middleware\require_advanced_feature;
use core\webapi\middleware\require_authenticated_user;
use core\webapi\mutation_resolver;
use mod_approval\exception\access_denied_exception;
use mod_approval\interactor\application_interactor;
use mod_approval\model\application\application;
use mod_approval\model\assignment\assignment_resolver;
use mod_approval\webapi\middleware\require_assignment;
use totara_job\entity\job_assignment;

/**
 * application_clone mutation.
 */
final class application_clone extends mutation_resolver {
    /**
     * @inheritDoc
     */
    public static function resolve(array $args, execution_context $ec): array {
        /** @var application $application */
        $application = $args['application'];
        $input = $args['input'];

        $user = user::logged_in();

        $interactor = application_interactor::from_application_model($application, $user->id);
        if (!$interactor->can_clone()) {
            throw access_denied_exception::application('Cannot perform clone');
        }

        if (!isset($input['job_assignment_id'])) {
            $job_assignment = null;
        } else {
            // Check that applicant is allowed to create an application with job assignment.
            $assignment_resolver = new assignment_resolver($application->user, $user);
            $assignment_resolver->resolve();
            $allowed_assignments = $assignment_resolver->get_menu_items();
            if (!$allowed_assignments->has('job_assignment_id', $input['job_assignment_id'])) {
                throw access_denied_exception::assignment('job assignment not available for this applicant');
            }
            $job_assignment = new job_assignment($input['job_assignment_id']);
        }

        $application = $application->clone($user->id, $job_assignment);

        if (!$ec->has_relevant_context()) {
            $ec->set_relevant_context($application->get_context());
        }
        return ['application' => $application];
    }

    /**
     * @inheritDoc
     */
    public static function get_middleware(): array {
        return [
            new require_advanced_feature('approval_workflows'),
            new require_authenticated_user(),
            require_assignment::by_input_application_id(),
        ];
    }
}
