<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2021 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Oleg Demeshev <oleg.demeshev@totaralearning.com>
 * @package mod_approval
 */

use core\json_editor\node\paragraph;
use core\json_editor\helper\document_helper;
use mod_approval\model\workflow\workflow_stage_approval_level;
use mod_approval\testing\generator as approval_generator;
use mod_approval\entity\workflow\workflow_version as workflow_version_entity;
use mod_approval\model\assignment\assignment_type;
use mod_approval\model\status;
use mod_approval\model\workflow\stage_type\form_submission;
use mod_approval\model\workflow\workflow;
use mod_approval\model\workflow\workflow_version;
use mod_approval\model\workflow\helper\cloner as workflow_clone_helper;
use mod_approval\testing\approval_workflow_test_setup;
use mod_approval\totara_notification\recipient\applicant_manager;
use mod_approval\totara_notification\resolver\level_started as level_started_resolver;
use totara_core\extended_context;
use totara_notification\loader\notification_preference_loader;
use totara_notification\testing\generator as notification_generator;
use totara_notification\json_editor\node\placeholder;

require_once(__DIR__ . '/totara_notification_level_base.php');

/**
 * @group approval_workflow
 * @coversDefaultClass mod_approval\model\workflow\helper\cloner
 */
class mod_approval_workflow_clone_test extends mod_approval_totara_notification_level_base_testcase {
    use approval_workflow_test_setup;

    /**
     * Gets the approval workflow generator instance
     *
     * @return approval_generator
     */
    protected function generator(): approval_generator {
        return approval_generator::instance();
    }

    public function test_clone_workflow(): void {
        global $DB;

        $now = time();
        $workflow = $this->set_application();
        $this->setAdminUser();

        // Change created and modified time for original container
        $container_record = $DB->get_record('course', ['id' => $workflow->course_id]);
        $container_record->timecreated = $now - HOURSECS;
        $container_record->timemodified = $now - HOURSECS;
        $DB->update_record('course', $container_record);

        $new_workflow_name = 'Learning to build';
        $new_default_assignment =  [
            'type' => assignment_type\cohort::get_code(),
            'id' => $this->getDataGenerator()->create_cohort()->id,
        ];

        $updated_workflow = $workflow->refresh();
        $new_workflow = workflow_clone_helper::clone(
            $updated_workflow,
            $new_workflow_name,
            $new_default_assignment['type'],
            $new_default_assignment['id']
        );

        $this->assertNotEmpty($new_workflow);
        $this->assertNotEquals($workflow->id, $new_workflow->id);
        $this->assertGreaterThan($workflow->id, $new_workflow->id);
        $this->assertEquals($new_workflow_name, $new_workflow->name);
        $this->assertEquals($workflow->workflow_type_id, $new_workflow->workflow_type_id);
        $this->assertEquals($workflow->latest_version->stages->first()->name, $new_workflow->latest_version->stages->first()->name);
        $this->assertEquals($workflow->latest_version->stages->first()->get_interactions()->count(), $new_workflow->latest_version->stages->first()->get_interactions()->count());
        $this->assertEquals($workflow->latest_version->stages->first()->get_formviews()->count(), $new_workflow->latest_version->stages->first()->get_formviews()->count());
        $this->assertNotEquals($workflow->course_id, $new_workflow->course_id);
        $this->assertGreaterThan($workflow->course_id, $new_workflow->course_id);
        $this->assertCount(1, $new_workflow->versions);
        $this->assertTrue($new_workflow->active);
        $this->assertEquals(status::DRAFT, $new_workflow->latest_version->status);
        // New course container has different timecreated and timemodified
        $this->assertNotEquals($updated_workflow->container->timecreated, $new_workflow->container->timecreated);
        $this->assertNotEquals($updated_workflow->container->timemodified, $new_workflow->container->timemodified);
        $this->assertEquals($new_default_assignment['type'], $new_workflow->default_assignment->assignment_type);
        $this->assertEquals($new_default_assignment['id'], $new_workflow->default_assignment->assignment_identifier);
    }

    public function test_clone_workflow_with_custom_notification(): void {
        $data = $this->setup_applications();
        $workflow = workflow::load_by_entity($data->workflow);
        $stage_1 = $workflow->latest_version->stages->first();
        $stage_2 = $workflow->latest_version->get_next_stage($stage_1->id);
        $stage_2_approval_level_1 = $stage_2->approval_levels->first();

        $extended_context = extended_context::make_with_context(
            $workflow->get_context(),
            'mod_approval',
            'workflow_stage',
            $stage_2->id
        );

        // Create a custom notification in workflow stage context.
        $notification_generator = notification_generator::instance();
        $notification_generator->create_notification_preference(
            level_started_resolver::class,
            $extended_context,
            [
                'schedule_offset' => 0,
                'recipient' => applicant_manager::class,
                'recipients' => [applicant_manager::class],
                'body_format' => FORMAT_JSON_EDITOR,
                'body' => document_helper::json_encode_document(
                    document_helper::create_document_from_content_nodes([
                        paragraph::create_json_node_from_text('Test notification body'),
                        paragraph::create_json_node_with_content_nodes([
                            placeholder::create_node_from_key_and_label('recipient:last_name', 'Recipient last name'),
                            placeholder::create_node_from_key_and_label('applicant:full_name', 'Applicant full name'),
                            placeholder::create_node_from_key_and_label('application:title', 'Application title'),
                            placeholder::create_node_from_key_and_label(
                                'applicant_job_assignment_organisation:full_name',
                                'Organisation full name'
                            ),
                            placeholder::create_node_from_key_and_label(
                                'workflow_stage:name',
                                'Workflow stage name'
                            ),
                            placeholder::create_node_from_key_and_label(
                                'approval_level:name',
                                'Approval level name'
                            ),
                        ]),
                    ])
                ),
                'additional_criteria' => json_encode([
                    'approval_level_id' => $stage_2_approval_level_1->id,
                ], JSON_THROW_ON_ERROR),
                'subject' => 'Test notification subject',
                'subject_format' => FORMAT_PLAIN,
                'enabled' => true,
            ]
        );

        $new_workflow_name = $workflow->name . ' Copy';
        $new_default_assignment =  [
            'type' => assignment_type\cohort::get_code(),
            'id' => static::getDataGenerator()->create_cohort()->id,
        ];

        $updated_workflow = $workflow->refresh();
        $new_workflow = workflow_clone_helper::clone(
            $updated_workflow,
            $new_workflow_name,
            $new_default_assignment['type'],
            $new_default_assignment['id']
        );

        static::assertNotEmpty($new_workflow);

        $new_workflow_version = $new_workflow->get_latest_version();
        $new_workflow_stages = $new_workflow_version->get_stages();
        foreach ($new_workflow_stages as $new_workflow_stage) {
            $new_extended_context = extended_context::make_with_id(
                $new_workflow->get_context()->id,
                'mod_approval',
                'workflow_stage',
                $new_workflow_stage->id
            );

            $notification_preferences = notification_preference_loader::get_notification_preferences($new_extended_context, null, true);
            foreach ($notification_preferences as $notification_preference) {
                $resolver_class_name = $notification_preference->get_resolver_class_name();
                if ($resolver_class_name !== level_started_resolver::class) {
                    // We only have custom notifications with level_started_resolver in our test.
                    continue;
                }
                $additional_criteria = $notification_preference->get_additional_criteria();
                if (!is_null($additional_criteria)) {
                    $criteria = (object)json_decode(
                        $additional_criteria,
                        true,
                        32,
                        JSON_THROW_ON_ERROR | JSON_INVALID_UTF8_SUBSTITUTE | JSON_BIGINT_AS_STRING

                    );
                    $approval_level_name = (workflow_stage_approval_level::load_by_id($criteria->approval_level_id))->name;
                    static::assertEquals($stage_2_approval_level_1->name, $approval_level_name);
                } else {
                    // Empty level means "All levels".
                }
            }
        }
    }

    private function set_application() {
        $this->setAdminUser();
        list($workflow) = $this->create_workflow_and_assignment();
        $user = $this->getDataGenerator()->create_user();
        $this->setUser($user);

        // Add a second approval level
        /** @var \mod_approval\entity\workflow\workflow $workflow */
        $workflow_version = workflow_version::load_latest_by_workflow_id($workflow->id);
        workflow_version_entity::repository()
            ->where('id', $workflow_version->id)
            ->update([
                'status' => status::DRAFT,
            ]);
        $stage_1 = $workflow_version->stages->first();
        $this->generator()->create_approval_level($workflow_version->get_next_stage($stage_1->id)->id, 'Level 2', 2);

        // Add a second stage
        $this->generator()->create_workflow_stage($workflow_version->id, 'Next Stage', form_submission::get_enum());
        workflow_version_entity::repository()
            ->where('id', $workflow_version->id)
            ->update([
                'status' => status::ACTIVE,
            ]);

        return workflow::load_by_id($workflow->id);
    }
}
