<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package mod_facetoface
 */

//
// Capability definitions for the facetoface module.
//
// The capabilities are loaded into the database table when the module is
// installed or updated. Whenever the capability definitions are updated,
// the module version number should be bumped up.
//
// The system has four possible values for a capability:
// CAP_ALLOW, CAP_PREVENT, CAP_PROHIBIT, and inherit (not set).
//
//
// CAPABILITY NAMING CONVENTION
//
// It is important that capability names are unique. The naming convention
// for capabilities that are specific to modules and blocks is as follows:
//   [mod/block]/<component_name>:<capabilityname>
//
// component_name should be the same as the directory name of the mod or block.
//
// Core moodle capabilities are defined thus:
//    moodle/<capabilityclass>:<capabilityname>
//
// Examples: mod/forum:viewpost
//           block/recent_activity:view
//           moodle/site:deleteuser
//
// The variable name for the capability definitions array follows the format
//   $<componenttype>_<component_name>_capabilities
//
// For the core capabilities, the variable is $moodle_capabilities.

$capabilities = array(

    // Ability to see that the activity exists, and the basic information.
    // about its sessions.
    'mod/facetoface:view' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'guest' => CAP_ALLOW,
            'student' => CAP_ALLOW,
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
            'apiuser' => CAP_ALLOW,
        )
    ),

    // Ability to signup for a session.
    'mod/facetoface:signup' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'student' => CAP_ALLOW,
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
            'apiuser' => CAP_ALLOW,
        )
    ),

    // Ability to see activities for which no sessions have been added.
    'mod/facetoface:viewemptyactivities' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // Ability to see how many people have signed up for a session.
    'mod/facetoface:viewattendees' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // Ability to take attendance.
    'mod/facetoface:takeattendance' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // Ability to add attendees to a session.
    'mod/facetoface:addattendees' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
        )
    ),

    // Ability to remove attendees from a session.
    'mod/facetoface:removeattendees' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
        )
    ),

    // Ability to manage archived attendees from a session.
    'mod/facetoface:managearchivedattendees' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            // No one is allowed by default.
        )
    ),

    // Ability to add, edit, copy and delete a session.
    'mod/facetoface:editevents' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:editsessions'
    ),

    // Ability to signup people on in progress and past events
    'mod/facetoface:signuppastevents' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:editevents'
    ),

    // Ability to add attendees outside sign-up registration period.
    'mod/facetoface:surpasssignupperiod' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:editevents'
    ),

    // View session cancellations.
    'mod/facetoface:viewcancellations' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    'mod/facetoface:configurecancellation' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    // Ability to overbook a session by signing up for it when wait list is disabled.
    'mod/facetoface:signupwaitlist' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:overbook'
    ),

    // Ability to create a f2f instance.
    'mod/facetoface:addinstance' => array(
        'riskbitmask' => RISK_XSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_COURSE,
        'archetypes' => array(
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'moodle/site:manageblocks'
    ),

    // Manager can reserve / allocate spaces for their team.
    'mod/facetoface:reservespace' => array(
        'riskbitmask' => RISK_SPAM,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'manager' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:addattendees'
    ),

    // Can reserve spaces on behalf of a manager.
    'mod/facetoface:reserveother' => array(
        'riskbitmask' => RISK_SPAM,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    // Can manage reservations for sessions.
    'mod/facetoface:managereservations' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    // Can approve any attendance
    'mod/facetoface:approveanyrequest' => array(
        'riskbitmask' => RISK_SPAM,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(),
    ),

    // Can view declare interest report.
    'mod/facetoface:viewinterestreport' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // Can export session sign-in sheet.
    'mod/facetoface:exportsessionsigninsheet' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'staffmanager' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // Ability to edit the job assignment a learner has used to sign up to a session.
    'mod/facetoface:changesignedupjobassignment' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),

    // View session attendees note.
    'mod/facetoface:viewattendeesnote' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    // Add/update/delete session attendees note.
    'mod/facetoface:manageattendeesnote' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
    ),

    // Managing facetoface custom fields.
    'mod/facetoface:managecustomfield' => array(
        'riskbitmask'   => RISK_CONFIG | RISK_DATALOSS,
        'captype'       => 'write',
        'contextlevel'  => CONTEXT_SYSTEM,
        'archetypes' => array(
            'manager' => CAP_ALLOW
        ),
        'clonepermissionsfrom' => 'mod/facetoface:updatefacetofacecustomfield',
    ),

    // View all sessions.
    'mod/facetoface:viewallsessions' => array(
        'riskbitmask'   => RISK_PERSONAL,
        'captype'       => 'read',
        'contextlevel'  => CONTEXT_SYSTEM,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'apiuser' => CAP_ALLOW,
        ),
    ),

    // Ability to add the message's recipients
    'mod/facetoface:addrecipients' => array(
        'riskbitmask' => RISK_SPAM,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'manager' => CAP_ALLOW,
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:addattendees'
    ),

    // Ability to remove the message's recipients
    'mod/facetoface:removerecipients' => array(
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'manager' => CAP_ALLOW,
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        ),
        'clonepermissionsfrom' => 'mod/facetoface:removeattendees'
    ),

    // Ability to add, edit and delete the sitewide facilitators
    'mod/facetoface:managesitewidefacilitators' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_SYSTEM,
        'archetypes' => array(
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to add, edit and delete the adhoc facilitators
    'mod/facetoface:manageadhocfacilitators' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to add, edit and delete the sitewide rooms
    'mod/facetoface:managesitewiderooms' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_SYSTEM,
        'archetypes' => array(
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to add, edit and delete the adhoc rooms
    'mod/facetoface:manageadhocrooms' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to add, edit and delete the sitewide assets
    'mod/facetoface:managesitewideassets' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_SYSTEM,
        'archetypes' => array(
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to add, edit and delete the adhoc assets
    'mod/facetoface:manageadhocassets' => array(
        'riskbitmask' => RISK_CONFIG | RISK_DATALOSS,
        'captype' => 'write',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW
        ),
    ),

    // Ability to see 'Join now' room link.
    'mod/facetoface:joinanyvirtualroom' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW
        ),
    ),

    // Ability to export attendance.
    'mod/facetoface:exportattendance' => array(
        'captype' => 'read',
        'contextlevel' => CONTEXT_MODULE,
        'archetypes' => array(
            'teacher' => CAP_ALLOW,
            'editingteacher' => CAP_ALLOW,
            'manager' => CAP_ALLOW,
            'tenantdomainmanager' => CAP_ALLOW,
        )
    ),
);
