<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.
/**
 * Unit tests for (some of) mod/feedback/lib.php.
 *
 * @package    mod_feedback
 * @copyright  2016 Stephen Bourget
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
defined('MOODLE_INTERNAL') || die();
global $CFG;
require_once($CFG->dirroot . '/mod/feedback/lib.php');

/**
 * Unit tests for (some of) mod/feedback/lib.php.
 *
 * @copyright  2016 Stephen Bourget
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class mod_feedback_lib_test extends \core_phpunit\testcase {

    public function test_feedback_get_completion_progress() {
        global $DB;
        $user = $this->getDataGenerator()->create_user();
        $course = $this->getDataGenerator()->create_course();
        $feedback = $this->getDataGenerator()->create_module('feedback', ['course' => $course->id, 'completionsubmit' => '1']);
        list($course, $cm) = get_course_and_cm_from_cmid($feedback->cmid, 'feedback');
        // This is tragic, but the creation of items is hardwired through mforms, and this style is copied from
        // the existing event tests found in mod/feedback/tests/events_test.php
        $item = new stdClass();
        $item->feedback = $feedback->id;
        $item->name = 'test';
        $item->typ = 'numeric';
        $item->presentation = '0|0';
        $item->hasvalue = '1';
        $itemid = $DB->insert_record('feedback_item', $item);
        $this->getDataGenerator()->enrol_user($user->id, $course->id);

        $this->setUser($user);

        $feedbackcompletion = new mod_feedback_completion($feedback, $cm, $course->id);
        $items = $feedbackcompletion->get_items();
        $this->assertTrue($feedbackcompletion->is_open());
        $this->assertFalse($feedbackcompletion->is_already_submitted());
        $this->assertTrue($feedbackcompletion->can_complete());
        $this->assertFalse($feedbackcompletion->is_empty());
        $this->assertCount(1, $items);

        $this->assertSame(array(), feedback_get_completion_progress($cm, $user->id));

        $data = new stdClass;
        $data->{'numeric_'.$itemid} = '1';
        $feedbackcompletion->save_response_tmp($data);

        // Its just temp presently.
        $this->assertFalse($feedbackcompletion->is_already_submitted());
        $this->assertSame(array(), feedback_get_completion_progress($cm, $user->id));

        // And now commit the save.
        $feedbackcompletion->save_response();
        $this->assertTrue($feedbackcompletion->is_already_submitted());
        $this->assertTrue($feedbackcompletion->can_complete());
        $this->assertSame(array('Submitted'), feedback_get_completion_progress($cm, $user->id));
    }

    /**
     * Tests for mod_feedback_refresh_events.
     */
    public function test_feedback_refresh_events() {
        global $DB;
        $this->setAdminUser();

        $timeopen = time();
        $timeclose = time() + 86400;

        $course = $this->getDataGenerator()->create_course();
        $generator = $this->getDataGenerator()->get_plugin_generator('mod_feedback');
        $params['course'] = $course->id;
        $params['timeopen'] = $timeopen;
        $params['timeclose'] = $timeclose;
        $feedback = $generator->create_instance($params);
        $cm = get_coursemodule_from_instance('feedback', $feedback->id);
        $context = context_module::instance($cm->id);

        // Normal case, with existing course.
        $this->assertTrue(feedback_refresh_events($course->id));
        $eventparams = array('modulename' => 'feedback', 'instance' => $feedback->id, 'eventtype' => 'open');
        $openevent = $DB->get_record('event', $eventparams, '*', MUST_EXIST);
        $this->assertEquals($openevent->timestart, $timeopen);

        $eventparams = array('modulename' => 'feedback', 'instance' => $feedback->id, 'eventtype' => 'close');
        $closeevent = $DB->get_record('event', $eventparams, '*', MUST_EXIST);
        $this->assertEquals($closeevent->timestart, $timeclose);
        // In case the course ID is passed as a numeric string.
        $this->assertTrue(feedback_refresh_events('' . $course->id));
        // Course ID not provided.
        $this->assertTrue(feedback_refresh_events());
        $eventparams = array('modulename' => 'feedback');
        $events = $DB->get_records('event', $eventparams);
        foreach ($events as $event) {
            if ($event->modulename === 'feedback' && $event->instance === $feedback->id && $event->eventtype === 'open') {
                $this->assertEquals($event->timestart, $timeopen);
            }
            if ($event->modulename === 'feedback' && $event->instance === $feedback->id && $event->eventtype === 'close') {
                $this->assertEquals($event->timestart, $timeclose);
            }
        }
    }

    /**
     * Test check_updates_since callback.
     */
    public function test_check_updates_since() {
        global $DB;

        $this->setAdminUser();
        $course = $this->getDataGenerator()->create_course();

        // Create user.
        $student = self::getDataGenerator()->create_user();

        // User enrolment.
        $studentrole = $DB->get_record('role', array('shortname' => 'student'));
        $this->getDataGenerator()->enrol_user($student->id, $course->id, $studentrole->id, 'manual');

        $this->setCurrentTimeStart();
        $record = array(
            'course' => $course->id,
            'custom' => 0,
            'feedback' => 1,
        );
        $feedback = $this->getDataGenerator()->create_module('feedback', $record);
        $cm = get_coursemodule_from_instance('feedback', $feedback->id, $course->id);
        $cm = cm_info::create($cm);

        $this->setUser($student);
        // Check that upon creation, the updates are only about the new configuration created.
        $onehourago = time() - HOURSECS;
        $updates = feedback_check_updates_since($cm, $onehourago);
        foreach ($updates as $el => $val) {
            if ($el == 'configuration') {
                $this->assertTrue($val->updated);
                $this->assertTimeCurrent($val->timeupdated);
            } else {
                $this->assertFalse($val->updated);
            }
        }

        $record = [
            'feedback' => $feedback->id,
            'userid' => $student->id,
            'timemodified' => time(),
            'random_response' => 0,
            'anonymous_response' => FEEDBACK_ANONYMOUS_NO,
            'courseid' => $course->id,
        ];
        $DB->insert_record('feedback_completed', (object) $record);
        $DB->insert_record('feedback_completedtmp', (object) $record);

        // Check now for finished and unfinished attempts.
        $updates = feedback_check_updates_since($cm, $onehourago);
        $this->assertTrue($updates->attemptsunfinished->updated);
        $this->assertCount(1, $updates->attemptsunfinished->itemids);

        $this->assertTrue($updates->attemptsfinished->updated);
        $this->assertCount(1, $updates->attemptsfinished->itemids);
    }

    /**
     * Test that if a teacher (non editing) is not part of any group in separate group mode he will not receive notification emails.
     * @covers ::feedback_get_receivemail_users
     */
    public function test_feedback_get_receivemail_users(): void {
        $this->resetAfterTest();
        $generator = $this->getDataGenerator();
        $course = $generator->create_course(['groupmode' => SEPARATEGROUPS, 'groupmodeforce' => 1]);
        $group = $generator->create_group(['courseid' => $course->id, 'name' => 'group0']);

        $feedbackgenerator = $generator->get_plugin_generator('mod_feedback');
        $feedback = $feedbackgenerator->create_instance(['course' => $course->id, 'email_notification' => true]);
        $teacher = [];
        $data = [
            'teacher1' => 'teacher',
            'teacher2' => 'teacher',
            'teacher3' => 'editingteacher',
            'teacher4' => 'editingteacher',
        ];
        foreach ($data as $username => $role) {
            $teacher[$username] = $generator->create_user(['username' => $username]);
            $generator->enrol_user($teacher[$username]->id, $course->id, $role);
        }
        $generator->create_group_member([
            'groupid' => $group->id,
            'userid' => $teacher['teacher1']->id,
        ]);
        $generator->create_group_member([
            'groupid' => $group->id,
            'userid' => $teacher['teacher4']->id,
        ]);

        $usergroup = $group->id;
        // Non editing Teachers (teacher1) in a group should receive notification emails.
        // Editing teachers (teacher4), in a group should also receive notification emails.
        $teachersingroup = feedback_get_receivemail_users($feedback->cmid, $usergroup);
        $this->assertCount(2, $teachersingroup);
        $this->assertNotEmpty($teachersingroup[$teacher['teacher1']->id]);
        $this->assertNotEmpty($teachersingroup[$teacher['teacher4']->id]);
        // Here we should return only the editing teachers (teacher3 and 4) who have access to all groups.
        $teacherwithnogroup = feedback_get_receivemail_users($feedback->cmid);
        $this->assertCount(2, $teacherwithnogroup);
        $this->assertNotEmpty($teacherwithnogroup[$teacher['teacher3']->id]);
        $this->assertNotEmpty($teacherwithnogroup[$teacher['teacher4']->id]);
    }
}
