<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Valerii Kuznetsov <valerii.kuznetsov@totaralms.com>
 * @package totara
 * @subpackage totara_appraisal
 */

use core\notification;

require_once(__DIR__ . '/../../config.php');
require_once($CFG->libdir.'/adminlib.php');
require_once($CFG->dirroot . '/totara/appraisal/lib.php');
require_once($CFG->dirroot . '/totara/appraisal/appraisal_forms.php');

// Check if Appraisals are enabled.
appraisal::check_feature_enabled();

admin_externalpage_setup('manageappraisals');
$systemcontext = context_system::instance();
require_capability('totara/appraisal:managepageelements', $systemcontext);

$action = optional_param('action', '', PARAM_ACTION);
$appraisalid = optional_param('appraisalid', 0, PARAM_INT);
$id = optional_param('id', 0, PARAM_INT); // Stage id.
$stage = null;
if ($appraisalid < 1) {
    if ($id < 1) {
        throw new appraisal_exception('Stage not found', 23);
    }
    $stage = new appraisal_stage($id);
    $appraisalid = $stage->appraisalid;
}
$appraisal = new appraisal($appraisalid);
if (!$stage) {
    $stage = new appraisal_stage($id);
}
$isdraft = appraisal::is_draft($appraisal);
$isreadonly = appraisal::is_read_only();

$returnurl = new moodle_url('/totara/appraisal/stage.php', array('appraisalid' => $appraisalid));

switch ($action) {
    case 'edit':
        appraisal::read_only_debugging('Create and update stage of legacy appraisal');
        $stage = new appraisal_stage($id);
        $defaults = $stage->get();
        $defaults->appraisalid = $appraisalid;
        $defaults->descriptionformat = FORMAT_HTML;
        $defaults = file_prepare_standard_editor($defaults, 'description', $TEXTAREA_OPTIONS, $TEXTAREA_OPTIONS['context'],
            'totara_appraisal', 'appraisal_stage', $id);
        $mform = new appraisal_stage_edit_form(null, array('action'=>$action, 'stage' => $defaults, 'readonly' => !$isdraft || $isreadonly));
        if ($mform->is_cancelled()) {
            redirect($returnurl);
        }
        if ($isdraft && $fromform = $mform->get_data() && !$isreadonly) {
            if (!confirm_sesskey()) {
                print_error('confirmsesskeybad', 'error');
            }
            if (empty($fromform->submitbutton)) {
                notification::error(get_string('error:unknownbuttonclicked', 'totara_appraisal'));
                redirect($returnurl);
            }
            if (!empty($fromform->timedue)) {
                // Set date to end-of-day.
                $fromform->timedue += ((int)$fromform->timedue > 0 ? (DAYSECS - 1) : 0);
                if ($fromform->timedue < time()) {
                    notification::error(get_string('error:completebyinvalid', 'totara_appraisal'));
                    break;
                }
            }
            $stage->set($fromform);
            if ($stage->id < 1) {
                $stage->save();
            }
            $fromform = file_postupdate_standard_editor($fromform, 'description', $TEXTAREA_OPTIONS, $TEXTAREA_OPTIONS['context'],
                'totara_appraisal', 'appraisal_stage', $stage->id);

            $stage->description = $fromform->description;
            $stage->save();

            // Create any pages that have been specified.
            if (isset($fromform->stageinitialpagetitles)) {
                $newpagenames = explode("\n", clean_param($fromform->stageinitialpagetitles, PARAM_TEXT));
                foreach ($newpagenames as $newpagename) {
                    if (trim($newpagename)) {
                        $newpage = new appraisal_page(0);
                        $todb = new stdClass();
                        $todb->appraisalstageid = $stage->id;
                        $todb->name = $newpagename;
                        $newpage->set($todb)->save();
                    }
                }
            }

            notification::success(get_string('stageupdated', 'totara_appraisal'));
            redirect($returnurl);
        }
        break;
    case 'delete':
        if ($stage->id < 1) {
            notification::error(get_string('error:stagenotfound', 'totara_appraisal'));
            redirect($returnurl);
        }
        if ($isdraft) {
            $confirm = optional_param('confirm', 0, PARAM_INT);
            if ($confirm == 1) {
                if (!confirm_sesskey()) {
                    print_error('confirmsesskeybad', 'error');
                }
                $stage->delete();
                notification::success(get_string('deletedstage', 'totara_appraisal'));
                redirect($returnurl);
            }
        } else {
            notification::error(get_string('error:appraisalmustdraft', 'totara_appraisal'));
            redirect($returnurl);
        }
        break;
    default:
        $stages = appraisal_stage::get_list($appraisalid);
        $pages = array();
        $stage = null;

        if ($stages && count($stages)) {
            if (!$id) {
                $id = current($stages)->id;
            }
            $stage = new appraisal_stage($id);

            if (isset($stage)) {
                $pages = appraisal_page::fetch_stage($stage->id);
                if ($stage->appraisalid != $appraisalid) {
                    throw new appraisal_exception('Stage must be within current appraisal');
                }
            }
        }
        break;
}

$output = $PAGE->get_renderer('totara_appraisal');
$title = $PAGE->title . ': ' . $appraisal->name;
$PAGE->set_title($title);
$PAGE->set_heading($appraisal->name);
$PAGE->navbar->add($appraisal->name);

$PAGE->requires->string_for_js('cancel', 'moodle');
$PAGE->requires->string_for_js('ok', 'moodle');
$PAGE->requires->string_for_js('yes', 'moodle');
$PAGE->requires->string_for_js('no', 'moodle');
$PAGE->requires->string_for_js('savechanges', 'moodle');
$PAGE->requires->string_for_js('confirmdeleteitem', 'totara_appraisal');
$PAGE->requires->string_for_js('confirmdeleteitemwithredisplay', 'totara_appraisal');
$PAGE->requires->string_for_js('error:cannotdelete', 'totara_appraisal');
$PAGE->requires->string_for_js('addpage', 'totara_appraisal');
$PAGE->requires->string_for_js('close', 'form');
$PAGE->requires->string_for_js('error:cannotdelete', 'totara_appraisal');

$jsmodule = array(
    'name' => 'totara_appraisal_stage',
    'fullpath' => '/totara/appraisal/js/stage.js',
    'requires' => array('json'));

$args = array('args' => '{"sesskey":"'.sesskey().'"}');
$PAGE->requires->js_init_call('M.totara_appraisal_stage.init', $args, false, $jsmodule);

echo $output->header();

appraisal::add_read_only_notification($isreadonly, $isdraft);

echo $output->heading(format_string($appraisal->name));
echo $output->appraisal_additional_actions($appraisal->status, $appraisal->id);

echo $output->appraisal_management_tabs($appraisal->id, 'content');

switch ($action) {
    case 'delete':
        echo $output->heading(get_string('deletestage', 'totara_appraisal', $stage->name),3);
        echo $output->confirm_delete_stage($stage->id);
        break;
    case 'edit':
        echo $mform->display();
        break;
    default:
        echo $output->heading(get_string('stages', 'totara_appraisal'), 3);
        echo $output->create_stage_button($appraisal);
        echo $output->appraisal_stages_table($stages, 0, $id);
        echo $output->stage_page_container($stage, $pages);
        break;
}

echo $output->footer();
