<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2020 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Tatsuhiro Kirihara <tatsuhiro.kirihara@totaralearning.com>
 * @package totara_msteams
 */

namespace totara_msteams;

defined('MOODLE_INTERNAL') || die;

use core\oauth2\client;
use moodle_exception;
use stdClass;
use totara_msteams\botfw\auth\jwt;
use totara_msteams\exception\auth_exception;

/**
 * An OpenID connect client built on top of the OAuth2 client.
 */
class oidcclient extends client {
    /**
     * Give it a very little clock-skew.
     */
    private const CLOCK_SKEW = 30; // 30 seconds

    /**
     * Is the user logged in? Note that if this is called
     * after the first part of the authorisation flow the token
     * is upgraded to an accesstoken.
     *
     * @return boolean true if logged in
     * @see \oauth2_client::is_logged_in()
     */
    public function is_logged_in_oidc(): bool {
        // Has the token expired?
        if (isset($this->accesstoken->expires) && time() >= $this->accesstoken->expires) {
            $this->log_out();
            return false;
        }

        // We have a token so we are logged in.
        if (isset($this->accesstoken->token)) {
            // Check that the access token has all the requested scopes.
            $scopemissing = false;
            $scopecheck = ' ' . $this->accesstoken->scope . ' ';

            $requiredscopes = explode(' ', $this->scope);
            foreach ($requiredscopes as $requiredscope) {
                if (strpos($scopecheck, ' ' . $requiredscope . ' ') === false) {
                    $scopemissing = true;
                    break;
                }
            }
            if (!$scopemissing) {
                return true;
            }
        }

        // If we've been passed then id_token generated by the
        // authorization server try and upgrade the token to an access token.
        $idtoken = optional_param('idtoken', null, PARAM_RAW);

        if ($idtoken && $this->upgrade_idtoken($idtoken)) {
            return true;
        }

        return false;
    }

    /**
     * Upgrade a authorization token from oauth 2.0 to an access token
     *
     * @param string $idtoken the code returned from the oauth authenticaiton
     * @return boolean true if token is upgraded successfully
     * @see \oauth2_client::upgrade_token()
     */
    public function upgrade_idtoken(string $idtoken): bool {
        $jwt = jwt::try_load($idtoken);
        if (!$jwt) {
            throw new auth_exception('id_token is malformed.');
        }

        $callbackurl = self::callback_url();
        $params = [
            'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            'scope' => 'user.read offline_access',
            'requested_token_use' => 'on_behalf_of',
            'assertion' => $idtoken,
            'redirect_uri' => $callbackurl->out(false),
        ];

        if ($this->basicauth) {
            $idsecret = urlencode($this->get_clientid()) . ':' . urlencode($this->get_clientsecret());
            $this->setHeader('Authorization: Basic ' . base64_encode($idsecret));
        } else {
            $params['client_id'] = $this->get_clientid();
            $params['client_secret'] = $this->get_clientsecret();
        }

        $time = time();

        // Requests can either use http GET or POST.
        if ($this->use_http_get()) {
            $response = $this->get($this->token_url(), $params);
        } else {
            $response = $this->post($this->token_url(), $this->build_post_data($params));
        }

        if ($this->info['http_code'] !== 200) {
            throw new moodle_exception('Could not upgrade oauth token');
        }

        $r = json_decode($response);

        if (is_null($r)) {
            throw new moodle_exception("Could not decode JSON token response");
        }

        if (!empty($r->error)) {
            throw new moodle_exception($r->error . ' ' . $r->error_description);
        }

        if (!isset($r->access_token)) {
            return false;
        }

        if (isset($r->refresh_token)) {
            $this->refreshtoken = $r->refresh_token;
        }

        // Store the token an expiry time.
        $accesstoken = new stdClass;
        $accesstoken->token = $r->access_token;
        if (isset($r->expires_in)) {
            $accesstoken->expires = $time + ($r->expires_in - self::CLOCK_SKEW);
        }
        $accesstoken->scope = $this->scope;
        $this->store_token($accesstoken);

        return true;
    }

    /**
     * @inheritDoc
     */
    final protected function use_http_get() {
        // HTTP GET is never permitted.
        return false;
    }
}
