<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2021 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author  Kian Nguyen <kian.nguyen@totaralearning.com>
 * @package totara_oauth2
 */

use core\orm\query\builder;
use core_phpunit\testcase;
use totara_api\model\client;
use totara_oauth2\entity\access_token;
use totara_oauth2\entity\client_provider;
use totara_oauth2\testing\generator;

/**
 * @group totara_oauth2
 */
class totara_oauth2_entity_client_provider_test extends testcase {
    /**
     * @return void
     */
    public function test_create_client_provider(): void {
        $db = builder::get_db();
        self::assertEquals(0, $db->count_records(client_provider::TABLE));

        $entity = new client_provider();
        $entity->client_id = "first_client_id";
        $entity->client_secret = "first_client_secret";
        $entity->name = "something else";
        $entity->internal = 1;
        $entity->client_secret_updated_at = time();
        $entity->save();

        self::assertEquals(1, $db->count_records(client_provider::TABLE));

        $record = $db->get_record(client_provider::TABLE, ["id" => $entity->id]);
        self::assertNotNull($record->time_created);
        self::assertEquals($record->client_id, $entity->client_id);
        self::assertEquals($record->client_secret, $entity->client_secret);
        self::assertEquals($record->name, $entity->name);
        self::assertEquals($record->internal, $entity->internal);
        self::assertEquals($record->client_secret_updated_at, $entity->client_secret_updated_at);

        self::assertNull($record->description);
        self::assertNull($record->description_format);
    }

    /**
     * @return void
     */
    public function test_delete_client_provider(): void {
        $db = builder::get_db();
        self::assertEquals(0, $db->count_records(client_provider::TABLE));

        $entity = new client_provider();
        $entity->client_id = "first_client_id";
        $entity->client_secret = "first_client_secret";
        $entity->name = "something else";
        $entity->save();

        self::assertEquals(1, $db->count_records(client_provider::TABLE));
        self::assertTrue($entity->exists());

        $entity->delete();
        self::assertEquals(0, $db->count_records(client_provider::TABLE));
        self::assertFalse($entity->exists());
    }

    /**
     * @return void
     */
    public function test_upgrade_client_provider(): void {
        $entity = new client_provider();
        $entity->client_id = "first_client_id";
        $entity->client_secret = "first_client_secret";
        $entity->name = "something else";
        $entity->save();

        $db = builder::get_db();

        self::assertTrue($db->record_exists(client_provider::TABLE, ["client_id" => "first_client_id"]));
        self::assertFalse($db->record_exists(client_provider::TABLE, ["client_id" => "second_client_id"]));

        $entity->client_id = "second_client_id";
        $entity->save();

        self::assertFalse($db->record_exists(client_provider::TABLE, ["client_id" => "first_client_id"]));
        self::assertTrue($db->record_exists(client_provider::TABLE, ["client_id" => "second_client_id"]));

        self::assertTrue($db->record_exists_select(client_provider::TABLE, $db->sql_compare_text("client_secret") . " = ?", ["first_client_secret"]));
        self::assertFalse($db->record_exists_select(client_provider::TABLE, $db->sql_compare_text("client_secret") . " = ?", ["second_client_secret"]));

        $entity->client_secret = "second_client_secret";
        $entity->save();

        self::assertFalse($db->record_exists_select(client_provider::TABLE, $db->sql_compare_text("client_secret") . " = ?", ["first_client_secret"]));
        self::assertTrue($db->record_exists_select(client_provider::TABLE, $db->sql_compare_text("client_secret") . " = ?", ["second_client_secret"]));
    }
    /**
     * @return void
     */
    public function test_clients_from_client_provider(): void {
        $generator = $this->getDataGenerator();
        $user = $generator->create_user();

        $db = builder::get_db();
        self::assertEquals(0, $db->count_records(client_provider::TABLE));
        $expect_client = client::create('test client', $user->id, '', null, true, ['create_client_provider' => true]);

        self::assertEquals(1, $db->count_records(client_provider::TABLE));
        $records = $db->get_records(client_provider::TABLE);
        $record = reset($records);

        $entity = new client_provider($record->id);
        $client = $entity->clients()->one();

        self::assertEquals($expect_client->id, $client->id);
    }

    /**
     * @return void
     * @throws dml_exception
     */
    public function test_access_tokens_from_client_provider(): void {
        $generator = generator::instance();

        $db = builder::get_db();
        $provider_1 = $generator->create_client_provider("client_id_one");
        $provider_2 = $generator->create_client_provider("client_id_two");
        $generator->create_access_token($provider_1->client_id);
        $generator->create_access_token($provider_2->client_id);

        self::assertEquals(2, $db->count_records(client_provider::TABLE));
        // There should be two access tokens - one for each provider
        self::assertEquals(2, $db->count_records(access_token::TABLE));

        $token_count = $provider_1->access_tokens()->count();

        self::assertEquals(1, $token_count);
    }
}
