<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Aaron Machin <aaron.machin@totara.com>
 * @package core_webservice
 */

namespace core_webservice\check\security;

defined('MOODLE_INTERNAL') || die();

global $CFG;

use coding_exception;
use core\check\check;
use core\check\result;

require_once($CFG->dirroot . '/webservice/lib.php');

/**
 * Checks whether insecure webservice protocols are enabled
 */
class insecure_protocols_enabled extends check {

    /**
     * @inheritDoc
     */
    public function get_name(): string {
        return get_string('check_insecure_protocols_enabled_name', 'webservice');
    }

    /**
     * @inheritDoc
     */
    public function get_action_link(): ?\action_link {
        return new \action_link(
            new \moodle_url('/admin/settings.php?section=webserviceprotocols'),
            get_string('actwebserviceshhdr', 'webservice'));
    }

    /**
     * @inheritDoc
     * @throws coding_exception
     */
    public function get_result(): result {
        // SOAP and XML-RPC are considered insecure as they transmit authentication secrets via GET parameters, which can be exposed in logs and URLs.
        $insecure_protocol_enabled = webservice_protocol_is_enabled('soap') || webservice_protocol_is_enabled('xmlrpc');

        if (!$insecure_protocol_enabled) {
            $status = result::OK;
            $summary = get_string('check_insecure_protocols_enabled_ok', 'webservice');
        } else {
            $status = result::WARNING;
            $summary = get_string('check_insecure_protocols_enabled_warning', 'webservice');
        }

        $details = get_string('check_insecure_protocols_enabled_details', 'webservice');

        return new result($status, $summary, $details);
    }
}

