<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Aaron Machin <aaron.machin@totara.com>
 * @package core_webservice
 */

use core\check\result;
use core_phpunit\testcase;
use core_webservice\check\security\insecure_protocols_enabled;

class core_webservice_insecure_protocols_enabled_test extends testcase {

    public static function webservice_protocols(): array {
        return [
            [['soap'], result::WARNING],
            [['soap,xmlrpc'], result::WARNING],
            [['soap,xmlrpc,rest'], result::WARNING],
            [['xmlrpc'], result::WARNING],
            [['rest,xmlrpc'], result::WARNING],
            [['rest,soap'], result::WARNING],
            [['rest'], result::OK],
        ];
    }

    /**
     * @dataProvider webservice_protocols
     * @return void
     * @throws coding_exception
     */
    public function test_webservice_is_disabled(array $protocols_to_enable) {
        set_config('enablewebservices', false);
        set_config('webserviceprotocols', implode(',', $protocols_to_enable));

        $check = new insecure_protocols_enabled();
        $result = $check->get_result();

        $this->assertEquals(result::OK, $result->get_status());
    }

    /**
     * @dataProvider webservice_protocols
     * @param array $protocols_to_enable
     * @param string $expected_status
     * @return void
     * @throws coding_exception
     */
    public function test_webservice_is_enabled(array $protocols_to_enable, string $expected_status) {
        set_config('enablewebservices', true);
        set_config('webserviceprotocols', implode(',', $protocols_to_enable));

        $check = new insecure_protocols_enabled();
        $result = $check->get_result();

        $this->assertEquals($expected_status, $result->get_status());
    }
}